#ifndef HIERARCHY_H
#define HIERARCHY_H


#include <iostream>
#include <sstream>
#include <fstream>
#include <math.h>
#include <stdlib.h>
#include <cstring>
#include <algorithm>
#include <deque>
#include <iomanip>
#include <cfloat>
#include "formcluster.h"
#include "buildgraph.h"
#include "graph.h"
#include "maxweight.h"

#ifndef ONETREE
#define ONETREE    1
#endif

#ifndef CLUSTERNUM
#define CLUSTERNUM 16
#endif

#ifndef LINELENGTH
#define LINELENGTH 1000
#endif

#ifndef DELTA
#define DELTA 0.0
#endif

#ifndef FACTOR
#define FACTOR 0.5
#endif

#ifndef MIN
#define MIN 0
#endif

using namespace std;

class Hierarchy
{
public:
    Hierarchy();

    void Begin(int);

    void Operation(int, int);

    void Operation(int, int, string);

    void Operation(int, int, int, string, int, int);

    void Split(int, int);

    void Combine(int, int, int);

    void ShowOneCluster(int);

    void Sub(int, int, int);

    void Clear();

    void merge_trees (vector<CClusterTree *> &trees,
                      float *modMatrix,
                      sorted_t &sorted,
                      float *a,
                      int nodeNum,
                      CMatrix &matrix,
                      int &clusterNum,
                      int &maxlevel);

    void merge_discon_tree(vector<CClusterTree *> &trees,
                           deque<int> &dq,
                           int &maxlevel,
                           float mod,
                           float a);

    bool force_Merge(vector<CClusterTree *> &trees, int num);

    void balance_tree(vector<CClusterTree *> &ttrees,
                      CClusterTree *&root,
                      float delta,
                      float factor);

    void split_tree(vector<CClusterTree *> &ttrees,
                    CClusterTree *&root,
                    CClusterTree *&subroot,
                    int partition_num,
                    float delta,
                    float factor);

    void rotate_tree(vector<CClusterTree *> &ttrees,
                     CClusterTree *&root,
                     CClusterTree *&node_a,
                     CClusterTree *node_b);


    /**
     *Acquire the modularity matrix to modMatrix
     */
    inline void initModMatrix(int* adjacencyMatrix, int nodeNum,
                float* modMatrix, int* degree, int edgeNum) {
        for (int i = 0; i < nodeNum; i++) {
            for (int j = i; j < nodeNum; j++) {
                if (adjacencyMatrix[i * nodeNum + j] == 1) {
                    modMatrix[2 * i * nodeNum + j] = 1.0f/(float)(2*edgeNum) -
                    (float)(degree[i]*degree[j])/(float)pow(2*edgeNum, 2);
                    modMatrix[2 * j * nodeNum + i] = 1.0f/(float)(2*edgeNum) -
                    (float)(degree[i]*degree[j])/(float)pow(2*edgeNum, 2);
                } else {
                    modMatrix[2 * i * nodeNum + j] = MIN;
                    modMatrix[2 * j * nodeNum + i] = MIN;
                }
            }
        }
    }

    inline void initModMatrixDirect(int* adjacencyMatrix, int nodeNum,
                float* modMatrix, int* outdegree, int* indegree, int edgeNum) {
        for (int i = 0; i < nodeNum; i++) {
            for (int j = 0; j < nodeNum; j++) {
                if (adjacencyMatrix[i * nodeNum + j] == 1) {
                    modMatrix[i * nodeNum + j] = 1.0f/(float)(2*edgeNum) -
                    (float)(outdegree[i]*indegree[j])/(float)pow(2*edgeNum, 2);
                } else {
                    modMatrix[i * nodeNum + j] = MIN;
                }
            }
        }
    }

    inline void initCluster(int nodeNum, cluster_t clusters) {
        for (int i = 0; i < nodeNum; i++) {
            clusters[i].push_back(i);
        }
    }

    inline void initializeA(int edgeNum, int* degree, float* a, int nodeNum) {
        for (int i = 0; i < nodeNum; i++)
            a[i] = (float)degree[i] / (float)(2 * edgeNum);
    }

    /**
     * This is for undirected graph.
     */
    inline void calculateDeg(int* adjacencyMatrix, int* degree, int nodeNum) {
        for (int i = 0; i < nodeNum; i++) {
            for (int j = 0; j < nodeNum; j++) {
                if (adjacencyMatrix[i * nodeNum + j] == 1)
                    degree[i] += 1;
            }
        }
    }

    /**
     * This is for directed graph.
     */
    inline void calculateDegDirect(int* adjacencyMatrix,
                                   int* outdegree,
                                   int* indegree,
                                   int nodeNum,
                                   int& edgeNum) {
        for (int i = 0; i < nodeNum; i++) {
            for (int j = 0; j < nodeNum; j++) {
                if (adjacencyMatrix[i*nodeNum+j] == 1) {
                    outdegree[i] += 1;
                    indegree[j] += 1;
                    edgeNum++;
                }
            }
        }
    }

    /**
     * This function revise the index into upper triangle matrix,
     * which means if x > y, we will revise it into x < y
     */
    inline void reviseIndex(int& x, int& y) {
        if (x > y) {
            int temp = y;
            y = x;
            x =temp;
        }
    }

    inline int calculateEdgeNum(int* degree, int nodeNum) {
        int m = 0;
        for (int i = 0; i < nodeNum; i++)
            m += degree[i];
        return m/2;
    }

    /**
     * Build a maximum heap for the whole matrix.
     * modMatrix: the matrix of modularity
     */
    inline void init_load_matrix(CMatrix& matrix,
                            float* modMatrix,
                            sorted_t &sorted,
                            int nodeNum) {
        matrix.setSize(nodeNum, nodeNum);
        matrix.Calloc();

        CPair* pPair = NULL;
        for (int x = 0; x < nodeNum; x++) {
            for (int y = x; y < nodeNum; y++) {
                if (modMatrix[x * nodeNum + y] != MIN) {
                    pPair = new CPair;

                    pPair->setA(x);
                    pPair->setB(y);
                    pPair->Iter(sorted.insert(
                                    sorted_t::value_type(
                                    modMatrix[x * nodeNum + y], pPair)));
                    matrix(x, y) = pPair;
                }
            }
        }
    }

    /**
     * Build a maximum heap for the whole matrix.
     * modMatrix: the matrix of modularity
     */
    inline void init_load_matrixDirec(CMatrix& matrix,
                            float* modMatrix,
                            sorted_t &sorted,
                            int nodeNum) {
        matrix.setSize(nodeNum, nodeNum);
        matrix.Calloc();

        CPair* pPair = NULL;
        for (int x = 0; x < nodeNum; x++) {
            for (int y = 0; y < nodeNum; y++) {
                if (modMatrix[x*nodeNum + y] != MIN) {
                    pPair = new CPair;

                    pPair->setA(x);
                    pPair->setB(y);
                    pPair->Iter(sorted.insert(sorted_t::value_type(
                                                  modMatrix[x*nodeNum + y],
                                              pPair)));
                    matrix(x, y) = pPair;
                }
            }
        }
    }

    inline void updateA(int cluster_num1,
                        int cluster_num2,
                        int new_id,
                        float* a)
    {
        a[new_id] = a[cluster_num2] + a[cluster_num1];
        a[cluster_num1] = 0;
        a[cluster_num2] = 0;
    }

    inline float getModularity(int nodeNum,
                               int index_x,
                               int index_y,
                               float* modMatrix)
    {
        return modMatrix[index_y * nodeNum + index_x];
    }

    inline void setModularity(int nodeNum,
                              int index_x,
                              int index_y,
                              float* modMatrix,
                              float modularity)
    {
        modMatrix[index_y * nodeNum + index_x] = modularity;
    }

    /**
     *merge cluster_num1 and cluster_num2 into new id
     */
    inline void updateModMatrix(float* modMatrix,
                                float* a,
                                int cluster_num1,
                                int cluster_num2,
                                int new_id,
                                int nodeNum,
                                bool show = false);

    inline void save_tree(CClusterTree *tree, ofstream &outf)
    {
        int index = -1;
        int id = -1;
        int level = -1;
        int size = 0;
        int count = 0;
        float mod = -1.0;
        float a = -1.0;
        map<int, float>::iterator mitr;
        vector<int>::iterator itr;
        set<int>::iterator sit;

        if (tree == NULL) {
            outf.write(reinterpret_cast<char *>(&level), sizeof(int));
            return;
        }

        /* tree level */
        level = tree->Level();
        outf.write(reinterpret_cast<char *>(&level), sizeof(int));

        /* tree id */
        id = tree->ID();
        outf.write(reinterpret_cast<char *>(&id), sizeof(int));

        /* tree index */
        index = tree->Index();
        outf.write(reinterpret_cast<char *>(&index), sizeof(int));

        /* leaf ids */
        size = tree->LeafIDs().size();
        outf.write(reinterpret_cast<char *>(&size), sizeof(int));
        for (sit = tree->LeafIDs().begin();
             sit != tree->LeafIDs().end();
             sit++)
        {
            id = *sit;
            outf.write(reinterpret_cast<char *>(&id), sizeof(int));
        }

        /* tree reserved modularity */
        count = 0;
        mitr = tree->getMods().begin();
        size = tree->getMods().size();
        outf.write(reinterpret_cast<char *>(&size), sizeof(int));
        while (count < size) {
            id = (*mitr).first;
            mod = (*mitr).second;
            outf.write(reinterpret_cast<char *>(&id), sizeof(int));
            outf.write(reinterpret_cast<char *>(&mod), sizeof(float));
            mitr++;
            count++;
        }

        /* tree reserved a */
        a = tree->getA();
        outf.write(reinterpret_cast<char *>(&a), sizeof(float));

        save_tree(tree->Child_L(), outf);
        save_tree(tree->Child_R(), outf);
    }

    inline void read_tree(CClusterTree * &tree,
                          ifstream &inf,
                          CClusterTree *parent = NULL)
    {
        int index = -1;
        int id = -1;
        int level = -1;
        int size = 0;
        int count = 0;
        float mod = -1.0;
        float a = -1.0;
        map<int, float>::iterator mitr;

        /* tree level */
        inf.read(reinterpret_cast<char *>(&level), sizeof(int));

        if (level == -1) {
            tree = NULL;
            return;
        }

        tree = new CClusterTree;
        assert(tree);

        tree->Level(level);

        /* parent */
        tree->Parent(parent);

        /* tree id */
        inf.read(reinterpret_cast<char *>(&id), sizeof(int));
        tree->ID(id);

        /* tree index */
        inf.read(reinterpret_cast<char *>(&index), sizeof(int));
        tree->Index(index);

        /* leaf ids */
        inf.read(reinterpret_cast<char *>(&size), sizeof(int));
        for (int i = 0; i < size; i++) {
            inf.read(reinterpret_cast<char *>(&id), sizeof(int));
            tree->LeafIDs().insert(id);
        }

        /* tree reserved modularity */
        count = 0;
        inf.read(reinterpret_cast<char *>(&size), sizeof(int));
        while (count < size) {
            inf.read(reinterpret_cast<char *>(&id), sizeof(int));
            inf.read(reinterpret_cast<char *>(&mod), sizeof(float));
            tree->setMod(id, mod);
            count++;
        }

        /* tree reserved a */
        inf.read(reinterpret_cast<char *>(&a), sizeof(float));
        tree->setA(a);

        read_tree(tree->Child_L(), inf, tree);
        read_tree(tree->Child_R(), inf, tree);
    }

    inline void init_trees(trees_t &trees,
                           int line_num,
                           float * mod,
                           float *a,
                           int nodeNum)
    {
        trees.clear();
        for (int i = 0; i < line_num; i++) {
            CClusterTree * tree = new CClusterTree;
            assert(tree);
            tree->Child_L(NULL);
            tree->Child_R(NULL);
            tree->ID(trees.size());
            tree->setA(a[i]); 
            tree->Level(0);
            tree->LeafIDs().insert(trees.size());

            assert(i == (int)(trees.size()));

            for (int j = 0; j < nodeNum; j++) {
                if (mod[i * nodeNum + j] != MIN) {
                    tree->setMod(j, mod[i * nodeNum + j]);
                }
            }
            trees.push_back(tree);
        }

        if (trees.size() != (unsigned int)line_num) {
            cerr << "the tree size is not equal to line number!" << endl;
            exit(0);
        }
    }

    inline void CollectClusterTree(CClusterTree *tree,
                                   int parent,
                                   int threshold,
                                   vector<int> &Clusters,
                                   vector<CClusterTree*> &ClusterTree)
    {
        if (tree == NULL || parent < threshold) {
            return;
        }

        if (tree->Level() <= threshold && parent > threshold)
        {
            Clusters.push_back(tree->ID());
            ClusterTree.push_back(tree);
        } else {
            CollectClusterTree(tree->Child_L(), tree->Level(), threshold,
                               Clusters, ClusterTree);
            CollectClusterTree(tree->Child_R(), tree->Level(), threshold,
                               Clusters, ClusterTree);
        }
    }

    inline void getClusteringRes(cluster_t clusters) {
        for (cluster_t::iterator iter = clusters.begin();
            iter != clusters.end(); iter++) {
            if ((*iter).second.size() != 0) {
                cout << "Hierarchical cluster " << (*iter).first << " : ";
                for (vector<int>::iterator iter_vec = (*iter).second.begin();
                    iter_vec != (*iter).second.end(); iter_vec++) {
                    cout << *iter_vec << " ";
                }
                cout<<endl;
            }
        }
        cout << "=====================================" << endl;
    }

    inline void setLabel(cluster_t clusters, GRAPH* g) {
        int label = 0;
        for (cluster_t::iterator iter = clusters.begin();
             iter != clusters.end(); iter++) {
            if ((*iter).second.size() != 0) {
                label = (*iter).first;
                for (vector<int>::iterator iter_vec = (*iter).second.begin();
                        iter_vec != (*iter).second.end(); iter_vec++) {
                    //g->adjList[*iter_vec + 1].vertex_label = label;
                    g->adjList[*iter_vec].vertex_id    = *iter_vec;
                    g->adjList[*iter_vec].vertex_label = (*iter).first;
                }
                //cout << endl;
            }
        }
    }

    inline void setLabel(cluster_t clusters, GRAPH* g, map<int, int> nodeMap) {
        //int label = 0;
        //int id = 0;
        nodeMap.clear();
        for (cluster_t::iterator iter = clusters.begin();
            iter != clusters.end(); iter++) {
            if ((*iter).second.size() != 0) {
                //label = (*iter).first;
                for (vector<int>::iterator iter_vec = (*iter).second.begin();
                        iter_vec != (*iter).second.end(); iter_vec++) {
                    //g->adjList[*iter_vec + 1].vertex_label = label;
                    g->adjList[*iter_vec].vertex_id    = *iter_vec;
                    g->adjList[*iter_vec].vertex_label = (*iter).first;
                }
            }
        }
    }

    inline CClusterTree* do_clustering(vector<CClusterTree *> &trees,
                       float* modMatrix,
                       sorted_t& sorted,
                       float* a,
                       int nodeNum,
                       CMatrix& matrix,
                       int& clusterNum,
                       int Num) {
        //int totalsize = trees.size();
        int maxlevel    = 0;
        float delta     = 0.0f;
        float factor    = 0.0f;

        m_nNewID = trees.size();
        while(clusterNum > Num && sorted.size() > 0) {
            merge_trees(trees, modMatrix, sorted, a, nodeNum, matrix,
                        clusterNum, maxlevel);
        }

        cout << "this is " << m_nCount << " time." << endl;
        bool merge = force_Merge(trees, ONETREE);
        CClusterTree *root = *(trees.rbegin());

        delta = DELTA;
        factor = FACTOR;
        if (delta > 0.0f) {
            balance_tree(trees, root, delta, factor);
        } else {
            cout << "Skip balancing" << endl;
        }
        m_nCount++;
        return root;
    }


    inline void max_heapify(deque<CClusterTree *> &A, int size, int id) {
        CClusterTree *tmp; 
        int largest;
        int l, r;
        l = 2 * id + 1;
        r = 2 * id + 2;
        if (l < size && A[l]->LeafNum() > A[id]->LeafNum()) {
            largest = l;
        } else {
            largest = id;
        }
        if (r < size && A[r]->LeafNum() > A[largest]->LeafNum())
        {
            largest = r;
        }
        if (largest != id) {
            //swap A[i] and A[largest] 
            tmp = A[id];
            A[id] = A[largest];
            A[largest] = tmp;
            max_heapify(A, size, largest);
        } 
    } 

    inline void build_max_heap(deque<CClusterTree *> &A, int size) {
        for (int i = size / 2; i >= 0; i--) {
            max_heapify(A, size, i);
        }
    }

    inline void tree_heapsort(deque<CClusterTree *> &A, int size) {
        int heap_size = size; 
        CClusterTree *tmp;
        build_max_heap(A, size);
        for(int i = size; i >= 1; i--) {
            //swap A[0] and A[i] 
            tmp = A[0];
            A[0] = A[i - 1];
            A[i - 1] = tmp;
            heap_size--;
            max_heapify(A, heap_size, 0); 
        }
    }

    inline deque<int> find_tree_id(deque<int> cluster_1, deque<int> cluster_2)
    {
        deque<int> dq_result;
        for (size_t i = 0; i < cluster_2.size(); i++) {
            for (size_t j = 0; j < cluster_1.size(); j++) {
                if (cluster_1[j] == cluster_2[i]) {
                    dq_result.push_back(cluster_2[i]);
                }
            }
        }
        return dq_result;
    }

    inline void traverse_tree(vector<CClusterTree *> &trees,
                              CClusterTree *&tree)
    {
        trees.push_back(tree);
        if (tree->Child_R() != NULL) {
            traverse_tree(trees, tree->Child_R());
        }

        if (tree->Child_L() != NULL) {
            traverse_tree(trees, tree->Child_L());
        }
    }

    inline void delete_tree(vector<CClusterTree *> &trees)
    {
        for (size_t i = 0; i < trees.size(); i++) {
            if (trees[i] != NULL) {
                delete trees[i];
                trees[i] = NULL;
            }
        }
    }

    inline void find_tree(vector<CClusterTree *> &trees,
                          CClusterTree *&tree,
                          int id)
    {
        assert(trees.size() != 0);
        for (size_t i = 0; i < trees.size(); i++) {
            if (trees[i]->ID() == id) {
                tree = trees[i];
                return;
            }
        }
    }

    inline bool vectorContain(vector<int> container, int id)
    {
        for (size_t i = 0; i < container.size(); i++) {
            if (container[i] == id) {
                return true;
            }
        }
        return false;
    }

    inline void DFS_Index(CClusterTree *tree, int &index)
    {
        tree->Index(index);
        index++;
        if (tree->Child_L() != NULL) {
            DFS_Index(tree->Child_L(), index);
        }
        if (tree->Child_R() != NULL) {
            DFS_Index(tree->Child_R(), index);
        }
    }

    inline void BFS_Reindex(CClusterTree *root)
    {
        int level = root->LeafNum() - 1;
        queue<CClusterTree *> q_trees;
        multimap<int, CClusterTree*, greater<int> > mm_Tree;
        assert(q_trees.size() == 0);
        q_trees.push(root);
        while (!q_trees.empty()) {
            if (q_trees.front()->Child_L() != NULL
                    && q_trees.front()->Child_R() != NULL) {
                mm_Tree.insert(pair<int, CClusterTree*>(q_trees.front()->Level(),
                                                        q_trees.front()));
                q_trees.push(q_trees.front()->Child_L());
                q_trees.push(q_trees.front()->Child_R());
            }
            q_trees.pop();
        }
        multimap<int, CClusterTree*, greater<int> >::iterator mitr;
        for (mitr = mm_Tree.begin(); mitr != mm_Tree.end(); mitr++) {
            mitr->second->Level(level);
            level--;
        }
    }

private:
    vector<GRAPH*>          graph;
    vector<int>             v_Diff;
    deque<int>              m_dqCluster;
    set<int>                m_sTreesID;
    cluster_t 				*m_tClusters;

    map<int, int>           *m_label;
    map<int, int>           map_ori_1;
    map<int, int>           map_ori_2;
    map<int, int>           map_1;
    map<int, int>           map_2;
    GRAPH                   *ori_g;
    BuildGraph              b;

    int                     *m_pNode;
    int                     m_nNewID;
    int                     m_nCount;
    int                     m_nTime;
};

#endif // HIERARCHY_H
