#include "glwidget.h"
#include "jitter.h"


void accFrustum(GLdouble left, GLdouble right, GLdouble bottom,
                GLdouble top, GLdouble near, GLdouble far, GLdouble pixdx,
                GLdouble pixdy, GLdouble eyedx, GLdouble eyedy, GLdouble focus)
{
    GLdouble xwsize, ywsize;
    GLdouble dx, dy;
    GLint viewport[4];

    glGetIntegerv (GL_VIEWPORT, viewport);

    xwsize = right - left;
    ywsize = top - bottom;

    dx = -(pixdx*xwsize/(GLdouble) viewport[2] + eyedx*near/focus);
    dy = -(pixdy*ywsize/(GLdouble) viewport[3] + eyedy*near/focus);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glFrustum (left + dx, right + dx, bottom + dy, top + dy, near, far);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslatef (-eyedx, -eyedy, 0.0);
}

void accPerspective(GLdouble fovy, GLdouble aspect,
                    GLdouble near, GLdouble far,
                    GLdouble pixdx, GLdouble pixdy,
                    GLdouble eyedx, GLdouble eyedy, GLdouble focus)
{
    double PI = 3.1415926;

    GLdouble fov2,left,right,bottom,top;

    fov2 = ((fovy*PI) / 180.0) / 2.0;

    top = near / (cos(fov2) / sin(fov2));
    bottom = -top;

    right = top * aspect;
    left = -right;

    accFrustum (left, right, bottom, top, near, far,
                pixdx, pixdy, eyedx, eyedy, focus);
}


GLWidget::GLWidget()
{

}

/**
 * @brief GLWidget::GLWidget
 * create the basic graph, initialize the openGL VBO
 */
GLWidget::GLWidget(float width, float height, float pad, QString infile_1,
                   QString infile_2, QWidget *parent, QGLWidget *shareWidget)
    : QGLWidget(parent, shareWidget)
{
    m_fThickness        = 0.0; // the thickness of the bar
    m_qSpan             = 0.0; // cardiograph span
    m_nInterval         = 27; // time interval
    m_nCluster          = 18; // cluster number
    m_nShowedCol        = m_nInterval; // the length of the time interval
//    m_fGlbMax           = 0.0;

    m_bCardioGraph      = false;
    m_bMouseClick       = false;
    m_bUserSelect       = false;
    m_bLineGraph        = false;
    m_bSubGraph         = false;

    m_bXSet             = true;
    m_bYSet             = true;
    m_bZSet             = true;
    m_bKSet             = true;
    m_bSSet             = true;
    m_bSubSet           = true;

    m_sInfile1          = infile_1;
    m_sInfile2          = infile_2;
    m_fWidth            = width;
    m_fHeight           = height;
    m_fPad              = pad;

    setMouseTracking(true);

    /**
     * initialize window size and position
     */
    resize(width, height);
    setWindowTitle("COMMGRAPH");

    /**
     * initialize the rendering area
     */
    m_rPanel = QRect(pad, pad, width - 2 * pad, height - 4 * pad);

    m_bUpdateBuffer     = true;
    m_pBarVert          = NULL;
    m_pBarVertColor     = NULL;
    m_pPolyVert         = NULL;
    m_pPlgVertColor     = NULL;
    m_pLineVert         = NULL;
    m_pLineVertColor    = NULL;
    m_pUBoundVert       = NULL;
    m_pUBoundVertColor  = NULL;
    m_pLBoundVert       = NULL;
    m_pLBoundVertColor  = NULL;

    Init();
    InitGraph();
    ShowPolygons();
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
}

/**
 *single polygon zoom in
 */
GLWidget::GLWidget(int width, int height, int pad, int cluster,
                   int sender, int recver,
                   QString infile_1, Hierarchy* &hierarchy, CMap* &map,
                   int time, string time1,
                   QWidget *parent, QGLWidget *shareWidget)
    : QGLWidget(parent, shareWidget)
{
    m_fThickness    = 0.0; // the thickness of the bar
    m_qSpan         = 0.0; // cardiograph span
    m_nInterval     = 1; // time interval
    m_nCluster      = 4320; // cluster number
    m_nShowedCol    = m_nInterval; // the length of the time interval
//    m_fGlbMax       = 0.0;

    m_bfullSub      = false;
    m_bMouseClick   = false;
    m_bUserSelect   = false;
    m_bCardioGraph  = false;
    m_bLineGraph    = false;
    m_bSubGraph     = true;

    m_bXSet         = true;
    m_bYSet         = true;
    m_bZSet         = true;
    m_bKSet         = true;
    m_bSSet         = true;
    m_bSubSet       = true;

    m_pHierarchy    = hierarchy;
    m_nSender       = sender;
    m_nRecver       = recver;
    m_sInfile1      = infile_1;
    m_nNorTime      = time;
    m_fHeight       = height;
    m_fWidth        = width;
    m_sTime         = time1;
    m_fPad          = pad;
    m_pMap          = map;
    m_nC            = cluster;

    setMouseTracking(true);

    /**
     * initialize window size and position
     */
    resize(width, height);
    setWindowTitle("SUBGRAPH");

    /**
     * initialize the rendering area
     */
    m_rPanel = QRect(pad, pad, width - 2 * pad, height - 4 * pad);

    m_bUpdateBuffer     = true;
    m_pBarVert          = NULL;
    m_pBarVertColor     = NULL;
    m_pPolyVert         = NULL;
    m_pPlgVertColor     = NULL;
    m_pLineVert         = NULL;
    m_pLineVertColor    = NULL;
    m_pUBoundVert       = NULL;
    m_pUBoundVertColor  = NULL;
    m_pLBoundVert       = NULL;
    m_pLBoundVertColor  = NULL;

    for (int i = 0; i < m_nInterval + 1; i++) {
        m_vDiff << 0.0;
    }

    //Init();
    InitGraph();
    ShowPolygons();
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
}

/**
 *full zoom in sub window
 */
GLWidget::GLWidget(int width, int height, int pad, QString infile_1,
                   Hierarchy* &hierarchy, CMap* &map, int time, string time1,
                   QWidget *parent, QGLWidget *shareWidget)
    : QGLWidget(parent, shareWidget)
{
    m_fThickness    = 0.0; // the thickness of the bar
    m_qSpan         = 0.0; // cardiograph span
    m_nInterval     = 1; // time interval
    m_nCluster      = 18; // cluster number
    m_nShowedCol    = 1; // the length of the time interval

    m_bfullSub      = true;
    m_bMouseClick   = false;
    m_bUserSelect   = false;
    m_bCardioGraph  = false;
    m_bLineGraph    = false;
    m_bSubGraph     = true;

    m_pHierarchy    = hierarchy;
    m_sInfile1      = infile_1;
    m_nNorTime      = time;
    m_fHeight       = height;
    m_fWidth        = width;
    m_sTime         = time1;
    m_fPad          = pad;
    m_pMap          = map;

    setMouseTracking(true);

    /**
     * initialize window size and position
     */
    resize(width, height);
    setWindowTitle("SUBGRAPH");

    /**
     * initialize the rendering area
     */
    m_rPanel = QRect(pad, pad, width - 2 * pad, height - 4 * pad);

    m_bUpdateBuffer     = true;
    m_pBarVert          = NULL;
    m_pBarVertColor     = NULL;
    m_pPolyVert         = NULL;
    m_pPlgVertColor     = NULL;
    m_pLineVert         = NULL;
    m_pLineVertColor    = NULL;
    m_pUBoundVert       = NULL;
    m_pUBoundVertColor  = NULL;
    m_pLBoundVert       = NULL;
    m_pLBoundVertColor  = NULL;

    for (int i = 0; i < m_nInterval + 1; i++) {
        m_vDiff << 0.0;
    }

    //Init();
    InitGraph();
    ShowPolygons();
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
}

/**
 * compute the buffer and size
 */
void GLWidget::UpdateBuffer()
{
    /**
     * initialize the bar vertices for the VBO
     */
    int n_BVNum = ((m_nCol + 1) * m_nRow) * m_pBars[0].Polygon().size();
    int n_BarNum = (m_nCol + 1) * m_nRow;

    if (m_pBarVert != NULL) {
        delete [] m_pBarVert;
    }
    if (m_pBarVertColor != NULL) {
        delete [] m_pBarVertColor;
    }

    m_pBarVert = new CVert[n_BVNum];
    m_pBarVertColor = new CVColor[n_BVNum];

    adjSize(m_nMax);

    for (int i = 0; i < n_BarNum; i++) {
        for (int j = 0; j < m_pBars[0].Polygon().size(); j++) {
            m_pBarVert[m_pBars[0].Polygon().size() * i + j].x
                    = (m_pBars[i].Polygon().data() + j)->x();
            m_pBarVert[m_pBars[0].Polygon().size() * i + j].y
                    = this->height() - (m_pBars[i].Polygon().data() + j)->y();
            m_pBarVertColor[m_pBars[0].Polygon().size() * i + j].red
                    = m_pBars[i].Color().redF();
            m_pBarVertColor[m_pBars[0].Polygon().size() * i + j].green
                    = m_pBars[i].Color().greenF();
            m_pBarVertColor[m_pBars[0].Polygon().size() * i + j].blue
                    = m_pBars[i].Color().blueF();
            m_pBarVertColor[m_pBars[0].Polygon().size() * i + j].alphaF
                    = m_pBars[i].Color().alphaF();
        }
    }

    /**
     * adjustable parameter
     */
    float f_glbMax      = 0.4;
    float f_glbMin      = 0.1;
    float f_glbAlpha    = 0.0;
    //float f_norMax      = 1;
    //float f_norMin      = 0.2;
    //float f_alpha       = 0.0;
    //float max_value = m_pData->MaxAmount();
    float max_value = m_pData->GlbMaxAmount();
    //m_fGlbMax = max_value > m_fGlbMax ? max_value : m_fGlbMax;
    int n_PVNum = m_nPTNum * 3;
    QColor q_PlgColor;

    if (m_pPolyVert != NULL) {
        delete [] m_pPolyVert;
    }
    if (m_pPlgVertColor != NULL) {
        delete [] m_pPlgVertColor;
    }

    /**
     * initialize the polygon vertices for the VBO
     */
    m_pPolyVert = new CVert[n_PVNum];
    m_pPlgVertColor = new CVColor[n_PVNum];

    for (int i = 0; i < m_pRelations.size(); i++) {
        for (int j = 0; j < m_pPolygon[i].getPolygon().size(); j++) {
            for (int k = 0; k < m_pPolygon[i].getPolygon().value(j).size(); k++)
            {
                q_PlgColor = m_pPolygon[i].getColor_1();
                f_glbAlpha = (m_pRelations[i].Amount() / max_value)
                             * (f_glbMax - f_glbMin) + f_glbMin;

                /*
                f_alpha = ((m_pPolygon[i].getWidth(j)
                            - m_pPolygon[i].getPlgMin())
                           / (m_pPolygon[i].getPlgMax()
                              - m_pPolygon[i].getPlgMin()))
                        * (f_norMax - f_norMin) + f_norMin;
                */

                m_pPolyVert[2 * PARTICLE * 3 * i + 3 * j + k].x
                        = (m_pPolygon[i].getPolygon().value(j).data() + k)->x();
                m_pPolyVert[2 * PARTICLE * 3 * i + 3 * j + k].y
                        = this->height()
                        - (m_pPolygon[i].getPolygon().value(j).data() + k)->y();
                m_pPlgVertColor[2 * PARTICLE * 3 * i + 3 * j + k].red
                                = q_PlgColor.redF();
                m_pPlgVertColor[2 * PARTICLE * 3 * i + 3 * j + k].green
                                = q_PlgColor.greenF();
                m_pPlgVertColor[2 * PARTICLE * 3 * i + 3 * j + k].blue
                                = q_PlgColor.blueF();
                m_pPlgVertColor[2 * PARTICLE * 3 * i + 3 * j + k].alphaF
                                = f_glbAlpha;
                //m_pPlgVertColor[200 * 3 * i + 3 * j + k].alphaF = f_alpha;
            }
        }
    }

    //int n_LineNum = (m_nCol + 1) * m_nRow;
    //int n_LinePNum = n_LineNum * 100;

    /**
     * inititalize the line for the VBO
     */
    int n_LineNum = m_pRelations.size();
    int n_LinePNum = n_LineNum * (PARTICLE + 1);

    if (m_pLineVert != NULL) {
        delete [] m_pLineVert;
    }
    if (m_pLineVertColor != NULL) {
        delete [] m_pLineVertColor;
    }
    m_pLineVert = new CVert[n_LinePNum];
    m_pLineVertColor = new CVColor[n_LinePNum];

    for (int i = 0; i < m_pRelations.size(); i++) {
        for (int j = 0; j <= PARTICLE; j++) {
            m_pLineVert[(PARTICLE + 1) * i + j].x
                    = m_pLines[i].getPath().pointAtPercent(j * 1.0
                                                           / PARTICLE).x();
            m_pLineVert[(PARTICLE + 1) * i + j].y = this->height()
                    - m_pLines[i].getPath().pointAtPercent(j * 1.0
                                                           / PARTICLE).y();
            m_pLineVertColor[(PARTICLE + 1) * i + j].red
                    = m_pPolygon[i].getColor_1().redF();
            m_pLineVertColor[(PARTICLE + 1) * i + j].green
                    = m_pPolygon[i].getColor_1().greenF();
            m_pLineVertColor[(PARTICLE + 1) * i + j].blue
                    = m_pPolygon[i].getColor_1().blueF();
            m_pLineVertColor[(PARTICLE + 1) * i + j].alphaF = .2f;
        }
    }

    if (m_pUBoundVert != NULL) {
        delete [] m_pUBoundVert;
    }
    if (m_pUBoundVertColor != NULL) {
        delete [] m_pUBoundVertColor;
    }
    if (m_pLBoundVert != NULL) {
        delete [] m_pLBoundVert;
    }
    if (m_pLBoundVertColor != NULL) {
        delete [] m_pLBoundVertColor;
    }

    m_pUBoundVert = new CVert[n_LinePNum];
    m_pUBoundVertColor = new CVColor[n_LinePNum];
    m_pLBoundVert = new CVert[n_LinePNum];
    m_pLBoundVertColor = new CVColor[n_LinePNum];


    for (int i = 0; i < m_pRelations.size(); i++) {
        for (int j = 0; j <= PARTICLE; j++) {
            m_pUBoundVert[(PARTICLE + 1) * i + j].x
                  = m_pPolygon[i].getPath_1().pointAtPercent(j * 1.0
                                                             / PARTICLE).x();
            m_pUBoundVert[(PARTICLE + 1) * i + j].y = this->height()
                  - m_pPolygon[i].getPath_1().pointAtPercent(j * 1.0
                                                             / PARTICLE).y();
            m_pUBoundVertColor[(PARTICLE + 1) * i + j].red = 0.5f;
            m_pUBoundVertColor[(PARTICLE + 1) * i + j].green = 0.5f;
            m_pUBoundVertColor[(PARTICLE + 1) * i + j].blue = 0.5f;
            m_pUBoundVertColor[(PARTICLE + 1) * i + j].alphaF = 0.1f;

            m_pLBoundVert[(PARTICLE + 1) * i + j].x
                  = m_pPolygon[i].getPath_2().pointAtPercent(j * 1.0
                                                             / PARTICLE).x();
            m_pLBoundVert[(PARTICLE + 1) * i + j].y = this->height()
                  - m_pPolygon[i].getPath_2().pointAtPercent(j * 1.0
                                                             / PARTICLE).y();
            m_pLBoundVertColor[(PARTICLE + 1) * i + j].red = 0.5f;
            m_pLBoundVertColor[(PARTICLE + 1) * i + j].green = 0.5f;
            m_pLBoundVertColor[(PARTICLE + 1) * i + j].blue = 0.5f;
            m_pLBoundVertColor[(PARTICLE + 1) * i + j].alphaF = 0.1f;
        }
    }
}

/**
 *compute bar position and size
 */
void GLWidget::UpdateBarPos()
{
    qreal qr_AdjSize;
    qreal q_thickness;
    qreal q_xPoint;
    qreal q_yPoint;
    float f_Send;
    float f_Recv;
    float f_adjSizeXmax = 0.0f;
    float f_adjSizeXmin = 0.0f;

    adjSize(m_nMax);

    for (int x = 0; x <= m_nCol; x++) {
        qr_AdjSize = m_fAdjSize * m_pData->getColMax(x) / m_pData->MaxAmount();
        q_yPoint = m_fHalfPanel - (m_fFixedSize * m_nRow + qr_AdjSize) / 2.0
                                   + m_vDiff[x] * m_qSpan;
//        q_yPoint = m_fHalfPanel - (((m_fHeight - 2 * m_fPad) / (m_nRow + m_fThickness *
//                                                              (m_pData->getColMax(m_nMax)
//                                                               / m_pData->MaxAmount()))) *
//                (m_nRow + m_fThickness * (m_pData->getColMax(x) / m_pData->MaxAmount()))) / 2;
        for (int y = 0; y < m_nRow; y++) {
            int id = y + x * m_nRow;
            f_Send = m_fFixedSize + m_fAdjSize * m_pBars[id].getSendValue();
            f_Recv = m_fFixedSize + m_fAdjSize * m_pBars[id].getRecvValue();

            q_thickness = f_Send > f_Recv ? f_Send : f_Recv;
            q_xPoint = x * 1.0 / m_nShowedCol * m_rPanel.width()
                    + m_rPanel.left();

            m_pBars[id].setShow(true);

            QPointF q_Rpoint_1(q_xPoint, q_yPoint
                               + (q_thickness - f_Recv) / 2.0);
            QPointF q_Rpoint_2(q_xPoint, q_yPoint
                               + (q_thickness - f_Recv) / 2.0 + f_Recv);
            QPointF q_Spoint_1(q_xPoint + 10.0, q_yPoint
                               + (q_thickness - f_Send) / 2.0);
            QPointF q_Spoint_2(q_xPoint + 10.0, q_yPoint
                               + (q_thickness - f_Send) / 2.0 + f_Send);
            QPolygonF q_Polygon;
            q_Polygon << q_Rpoint_1 << q_Rpoint_2
                      << q_Spoint_2 << q_Spoint_1;
            m_pBars[id].Polygon(q_Polygon);
            q_yPoint = q_yPoint + q_thickness + 0;

            q_xPoint > f_adjSizeXmax ? f_adjSizeXmax = q_xPoint :
                    f_adjSizeXmax = f_adjSizeXmax;
            q_xPoint > f_adjSizeXmin ? f_adjSizeXmin = f_adjSizeXmin :
                    f_adjSizeXmin = q_xPoint;
        }
    }
    this->resize(f_adjSizeXmax + 10.0 + m_fPad, m_fHeight);
}

/**
 * compute line position
 */
void GLWidget::UpdateLinePos()
{
    QList<QPainterPath> q_pPath;
    QPointF q_pGStart, q_pGEnd;
    EBundle f_pBundle;
    qreal qr_AdjSize_1;
    qreal qr_AdjSize_2;
    qreal q_sPoint_1;
    qreal q_ePoint_1;
    qreal q_sPoint_2;
    qreal q_ePoint_2;
    qreal q_sPad;
    qreal q_ePad;

    adjSize(m_nMax);

    for (int i = 0; i < m_pRelations.size(); i++)
    {
        qr_AdjSize_1 = m_fAdjSize *
                       m_pData->getColMax(m_pRelations[i].TimeIndex())
                       / m_pData->MaxAmount();
        qr_AdjSize_2 = m_fAdjSize *
                       m_pData->getColMax(m_pRelations[i].TimeIndex() + 1)
                       / m_pData->MaxAmount();

        q_sPad = m_fHalfPanel - (m_fFixedSize * m_nRow + qr_AdjSize_1) / 2
                                 + m_vDiff[m_pRelations[i].TimeIndex()]
                                 * m_qSpan;

        q_ePad = m_fHalfPanel - (m_fFixedSize * m_nRow + qr_AdjSize_2) / 2
                                 + m_vDiff[m_pRelations[i].TimeIndex() + 1]
                                 * m_qSpan;

//        q_sPad = m_fHalfPanel - (((m_fHeight - 2 * m_fPad) / (m_nRow + m_fThickness *
//                                                              (m_pData->getColMax(m_nMax)
//                                                               / m_pData->MaxAmount()))) *
//                (m_nRow + m_fThickness * (m_pData->getColMax(m_pRelations[i].TimeIndex()) / m_pData->MaxAmount()))) / 2;

//        q_ePad = m_fHalfPanel - (((m_fHeight - 2 * m_fPad) / (m_nRow + m_fThickness *
//                                                              (m_pData->getColMax(m_nMax)
//                                                               / m_pData->MaxAmount()))) *
//                (m_nRow + m_fThickness * (m_pData->getColMax(m_pRelations[i].TimeIndex() + 1) / m_pData->MaxAmount()))) / 2;

        qreal q_S1 = (m_pRelations[i].SenderIndex() % m_nENum + 1)
                     * m_fFixedSize + m_fAdjSize
                     * m_pData->getAccValue(m_pRelations[i].SenderIndex()
                                            % m_nENum
                     + m_pRelations[i].TimeIndex() * m_nRow)
                     / m_pData->MaxAmount();
        qreal q_S2 = m_pRelations[i].SenderIndex() % m_nENum * m_fFixedSize
                     + m_fAdjSize *
                     m_pData->getAccValue(m_pRelations[i].SenderIndex()
                                          % m_nENum
                     + m_pRelations[i].TimeIndex() * m_nRow - 1)
                     / m_pData->MaxAmount();
        qreal q_Sdc = m_fFixedSize + m_fAdjSize
                      * m_pData->SendAmount(m_pRelations[i].SenderIndex()
                                            % m_nENum,
                      m_pRelations[i].TimeIndex()) / m_pData->MaxAmount();
        qreal q_R1 = (m_pRelations[i].ReceiverIndex() % m_nENum + 1)
                     * m_fFixedSize + m_fAdjSize *
                     m_pData->getAccValue(m_pRelations[i].ReceiverIndex()
                                          % m_nENum
                     + (m_pRelations[i].TimeIndex() + 1) * m_nRow)
                     / m_pData->MaxAmount();
        qreal q_R2 = m_pRelations[i].ReceiverIndex() % m_nENum * m_fFixedSize
                     + m_fAdjSize
                     * m_pData->getAccValue(m_pRelations[i].ReceiverIndex()
                                            % m_nENum
                     + (m_pRelations[i].TimeIndex() + 1) * m_nRow - 1)
                     / m_pData->MaxAmount();
        qreal q_Rdc = m_fFixedSize + m_fAdjSize
                      * m_pData->RecvAmount(m_pRelations[i].ReceiverIndex()
                                            % m_nENum,
                      m_pRelations[i].TimeIndex() + 1) / m_pData->MaxAmount();

        q_sPoint_1 = (q_S1 + q_S2 - q_Sdc) / 2 ;
        q_sPoint_2 = q_sPoint_1 + q_Sdc;
        q_ePoint_1 = (q_R1 + q_R2 - q_Rdc) / 2 ;
        q_ePoint_2 = q_ePoint_1 + q_Rdc;

        qreal q_Stmp = m_fFixedSize + m_fAdjSize
                       * m_pData->SendAmount(m_pRelations[i].SenderIndex()
                                             % m_nENum,
                       m_pRelations[i].TimeIndex()) / m_pData->MaxAmount();
        qreal q_Rtmp = m_fFixedSize + m_fAdjSize
                       * m_pData->RecvAmount(m_pRelations[i].SenderIndex()
                                             % m_nENum,
                       m_pRelations[i].TimeIndex()) / m_pData->MaxAmount();
        qreal q_Mtmp = q_Stmp > q_Rtmp ? q_Stmp : q_Rtmp;

        if (m_pRelations[i].SenderIndex() % m_nENum == 0) {
            q_sPoint_1 = (q_Mtmp - q_Sdc) / 2;
            q_sPoint_2 = q_sPoint_1 + q_Sdc;
        }

        qreal q_Stmp2 = m_fFixedSize + m_fAdjSize
                        * m_pData->SendAmount(m_pRelations[i].ReceiverIndex()
                                              % m_nENum,
                        m_pRelations[i].TimeIndex() + 1) / m_pData->MaxAmount();
        qreal q_Rtmp2 = m_fFixedSize + m_fAdjSize
                        * m_pData->RecvAmount(m_pRelations[i].ReceiverIndex()
                                              % m_nENum,
                        m_pRelations[i].TimeIndex() + 1) / m_pData->MaxAmount();
        qreal q_Mtmp2 = q_Stmp2 > q_Rtmp2 ? q_Stmp2 : q_Rtmp2;

        if (m_pRelations[i].ReceiverIndex() % m_nENum == 0) {
            q_ePoint_1 = (q_Mtmp2 - q_Rdc) / 2;
            q_ePoint_2 = q_ePoint_1 + q_Rdc;
        }

        qreal qf_pad1 = 0;
        QPointF q_pStart_1 (m_pRelations[i].TimeIndex() * 1.0
                            / m_nShowedCol * m_rPanel.width()
                            + m_rPanel.left() + 10,
                            q_sPad + q_sPoint_1 + qf_pad1);
        QPointF q_pEnd_1 ((m_pRelations[i].TimeIndex()+1) * 1.0
                          / m_nShowedCol * m_rPanel.width()
                          + m_rPanel.left(),
                          q_ePad + q_ePoint_1 + qf_pad1);
        QPointF q_pStart_2 (m_pRelations[i].TimeIndex() * 1.0
                            / m_nShowedCol * m_rPanel.width()
                            + m_rPanel.left() + 10,
                           q_sPad + q_sPoint_2 + qf_pad1);
        QPointF q_pEnd_2 ((m_pRelations[i].TimeIndex()+1) * 1.0
                          / m_nShowedCol * m_rPanel.width()
                          + m_rPanel.left(),
                          q_ePad + q_ePoint_2 + qf_pad1);

        qreal qf_pad2 = 0;

        QPointF a_1(q_pStart_1.x() + (1.0 / m_nShowedCol * m_rPanel.width() / 2
                    - 1.0 / m_nShowedCol * m_rPanel.width() / 4),
                    q_sPad + q_sPoint_1 + qf_pad2);
        QPointF b_1(q_pStart_1.x() + (1.0 / m_nShowedCol * m_rPanel.width() / 2
                    - 1.0 / m_nShowedCol * m_rPanel.width() / 8),
                    q_sPad + q_sPoint_1 + qf_pad2);
        QPointF d_1(q_pStart_1.x() + (1.0 / m_nShowedCol * m_rPanel.width() / 2
                    + 1.0 / m_nShowedCol * m_rPanel.width() / 4),
                    q_sPad + q_sPoint_1 + qf_pad2);
        QPointF m_1(q_pStart_1.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 2,
                    q_sPad + q_sPoint_1 + qf_pad2);

//        QPointF a_1(q_pStart_1.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 8,
//                    q_sPad + q_sPoint_1 + qf_pad2);
//        QPointF b_1(q_pStart_1.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 4,
//                    q_sPad + q_sPoint_1 + qf_pad2);
//        QPointF d_1(q_pStart_1.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 2,
//                    q_sPad + q_sPoint_1 + qf_pad2);
//        QPointF m_1(q_pStart_1.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 8 * 3,
//                    q_sPad + q_sPoint_1 + qf_pad2);

        QPointF a_2(q_pStart_2.x() + (1.0 / m_nShowedCol * m_rPanel.width() / 2
                    - 1.0 / m_nShowedCol * m_rPanel.width() / 4),
                    q_sPad + q_sPoint_2 + qf_pad2);
        QPointF b_2(q_pStart_2.x() + (1.0 / m_nShowedCol * m_rPanel.width() / 2
                    - 1.0 / m_nShowedCol * m_rPanel.width() / 8),
                    q_sPad + q_sPoint_2 + qf_pad2);
        QPointF d_2(q_pStart_2.x() + (1.0 / m_nShowedCol * m_rPanel.width() / 2
                    + 1.0 / m_nShowedCol * m_rPanel.width() / 4),
                    q_sPad + q_sPoint_2 + qf_pad2);
        QPointF m_2(q_pStart_2.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 2,
                    q_sPad + q_sPoint_2 + qf_pad2);

//        QPointF a_2(q_pStart_2.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 8,
//                    q_sPad + q_sPoint_2 + qf_pad2);
//        QPointF b_2(q_pStart_2.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 4,
//                    q_sPad + q_sPoint_2 + qf_pad2);
//        QPointF d_2(q_pStart_2.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 2,
//                    q_sPad + q_sPoint_2 + qf_pad2);
//        QPointF m_2(q_pStart_2.x() + 1.0 / m_nShowedCol * m_rPanel.width() / 8 * 3,
//                    q_sPad + q_sPoint_2 + qf_pad2);

        q_pGStart = (q_pStart_1+q_pStart_2) / 2;
        q_pGEnd = (q_pEnd_1+q_pEnd_2) / 2;

        q_pPath.push_back(f_pBundle.ctrl_Path(a_1, b_1, d_1, m_1,
                                              q_pStart_1, q_pEnd_1, 1.0));

        q_pPath.push_back(f_pBundle.ctrl_Path(a_2, b_2, d_2, m_2,
                                              q_pStart_2, q_pEnd_2, 1.0));

        f_pBundle.setPath(q_pPath[0], q_pPath[1]);

        m_pPolygon[i].gdtStart(q_pGStart);
        m_pPolygon[i].gdtEnd(q_pGEnd);
        m_pPolygon[i].setPath_1(q_pPath[0]);
        m_pPolygon[i].setPath_2(q_pPath[1]);
        m_pPolygon[i].setPolygon(f_pBundle.toPolygon(q_pPath));
        m_pPolygon[i].setPlg(f_pBundle.getPolygon());
        m_pPolygon[i].setWidth(f_pBundle.setWidth());
        m_pPolygon[i].setPlgMaxMin();
        m_pPolygon[i].setColor_1(m_pLines[i].Color_1());
        m_pPolygon[i].setColor_2(m_pLines[i].Color_2());
        m_pPolygon[i].setTime(m_pRelations[i].TimeIndex());
        //m_pPolygon[i].setSTime(m_pRelations[i].Time());
        m_pLines[i].setPath(f_pBundle.ctrl_Path((a_1+a_2)/2, (b_1+b_2)/2,
                                                (d_1+d_2)/2, (m_1+m_2)/2,
                                                (q_pStart_1+q_pStart_2)/2,
                                                (q_pEnd_1+q_pEnd_2)/2, 0.5));
        m_pLines[i].gdtStart(q_pGStart);
        m_pLines[i].gdtEnd(q_pGEnd);
        q_pPath.clear();

        m_pPolygon[i].m_vCtrlPoints.clear();
        m_pPolygon[i].m_vCtrlPoints.push_back(a_1);
        m_pPolygon[i].m_vCtrlPoints.push_back(b_1);
        m_pPolygon[i].m_vCtrlPoints.push_back(d_1);
        m_pPolygon[i].m_vCtrlPoints.push_back(m_1);
        m_pPolygon[i].m_vCtrlPoints.push_back(a_2);
        m_pPolygon[i].m_vCtrlPoints.push_back(b_2);
        m_pPolygon[i].m_vCtrlPoints.push_back(d_2);
        m_pPolygon[i].m_vCtrlPoints.push_back(m_2);



    }
    m_nPTNum = f_pBundle.getNum();
}

/**
 *compute label position and size
 */
void GLWidget::UpdateLabelPos()
{
    qreal q_Point;
    qreal q_thickness;
    qreal q_middle;
    qreal qr_AdjSize;
    float f_Send;
    float f_Recv;

    adjSize(m_nMax);

    for (int x = 0; x <= m_nCol; x++) {
        qr_AdjSize = m_fAdjSize * m_pData->getColMax(x) / m_pData->MaxAmount();
        q_Point = m_fHalfPanel - (m_fFixedSize * m_nRow + qr_AdjSize) / 2
                                  + m_vDiff[x] * m_qSpan;
        for (int y = 0; y < m_nRow; y++) {
            int id = y + x * m_nRow;
            f_Send = (m_fFixedSize + m_fAdjSize * m_pData->SendAmount(y, x)
                      / m_pData->MaxAmount()) / 2;
            f_Recv = (m_fFixedSize + m_fAdjSize * m_pData->RecvAmount(y, x)
                      / m_pData->MaxAmount()) / 2;
            q_middle = f_Send > f_Recv ? f_Send : f_Recv;
            if (y == 0)
            {
                q_thickness = 0.0;
            }
            else {
                q_thickness = m_fFixedSize * y
                              + m_fAdjSize
                              * m_pData->getAccValue(y - 1 + x * m_nRow)
                              / m_pData->MaxAmount();
            }
            m_pLabels[id].setShow(true);
            m_pLabels[id].setGeometry((x * 1.0 / m_nShowedCol
                                        * m_rPanel.width())
                                        + m_rPanel.left() - 18,
                                        (q_Point + q_thickness
                                         + q_middle + 2),
                                        20,
                                        10);

            if (f_Send == 0 && f_Recv ==0){
                m_pLabels[id].setShow(false);
            }
        }
    }
}

void GLWidget::UpdateTagPos()
{
    //qreal qr_LabelX, qr_LabelY;
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pMovePos,
                                                 Qt::OddEvenFill)) {
            //qr_LabelX = m_pMovePos.x();
            //qr_LabelY = m_pMovePos.y();
            m_sText = "   " + m_pRelations[i].Sender() + " sends "
                                + QString::number(m_pRelations[i].Amount())
                                + " to " + m_pRelations[i].Receiver()
                                + " at " + m_pRelations[i].Time();
            m_qWidthPad = 30 * sizeof(m_sText);
            m_qHeightPad = 16;
            return;
        }
        else {
            m_sText.clear();
            m_qWidthPad = 0;
            m_qHeightPad = 0;
        }
    }
    for (int i = 0; i < (m_nRow * (m_nCol + 1)); i++) {
        if (m_pBars[i].Polygon().containsPoint(m_pMovePos, Qt::OddEvenFill)) {
            //qr_LabelX = m_pMovePos.x();
            //qr_LabelY = m_pMovePos.y();
            m_sText = "   " + m_pLabels[i].getLabel();
            m_qWidthPad = 50;
            m_qHeightPad = 16;
            return;
        }
        else {
            m_sText.clear();
            m_qWidthPad = 0;
            m_qHeightPad = 0;
        }
    }
}

void GLWidget::UpdateSinglePolygonPos()
{
    m_nTransTime = -1;
    m_nTransIndex = -1;
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pPos, Qt::OddEvenFill)) {
            m_nTransIndex = i;
            m_nTransTime = m_pPolygon[i].getTime();
            m_cRelation = m_pRelations[i];
            m_pPolygon[i].setShow(true);
            break;
        }
    }
    m_bUpdateBuffer = true;
    repaint();
}

void GLWidget::UpdateMSelection()
{
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pPos, Qt::OddEvenFill)) {
            m_pPolygon[i].setShow(true);
                break;
        }
    }
}

void GLWidget::UpdatePlgBar_L()
{
    for (int x = 0; x <= m_nCol; x++) {
        for (int y = 0; y < m_nRow; y++) {
            int id = y + x * m_nRow;
            if (m_pBars[id].Polygon().containsPoint(m_pLDbClickPos,
                                                    Qt::OddEvenFill)) {
                QVector<int> qn_Send = m_pBars[id].popSend();
                for (int i = 0; i < qn_Send.size(); i++) {
                    m_pPolygon[qn_Send[i]].setShow(true);
                    m_pLines[qn_Send[i]].setShow(true);
                }
            }
        }
    }
}


void GLWidget::UpdatePlgBar_R()
{
    for (int x = 0; x <= m_nCol; x++) {
        for (int y = 0; y < m_nRow; y++) {
            int id = y + x * m_nRow;
            if (m_pBars[id].Polygon().containsPoint(m_pRDbClickPos,
                                                    Qt::OddEvenFill)) {
                QVector<int> qn_Recv = m_pBars[id].popRecv();
                for (int i = 0; i < qn_Recv.size(); i++) {
                    m_pPolygon[qn_Recv[i]].setShow(true);
                    m_pLines[qn_Recv[i]].setShow(true);
                }
            }
        }
    }
}

void GLWidget::initializeGL()
{
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glClearColor(1.0, 1.0, 1.0, 0);
}

void GLWidget::resizeGL(int w, int h)
{
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, w, 0, h, -10, 10);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glViewport(0, 0, w, h);
}

void GLWidget::paintGL()
{

    GLint viewport[4];
    glGetIntegerv(GL_VIEWPORT, viewport);

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    //glPushMatrix();

//    return;

    int nImageQuality = 8;
    int nLoop = 5;
    float dx, dy;

    switch(nImageQuality) {
        case 1: nLoop = 1; break;
        case 2: nLoop = 2; break;
        case 3: nLoop = 3; break;
        case 4: nLoop = 4; break;
        case 5: nLoop = 8; break;
        case 6: nLoop = 15; break;
        case 7: nLoop = 24; break;
        case 8: nLoop = 66; break;
        default:nLoop = 1;
    }

    //glMatrixMode(GL_PROJECTION);
    //glPushMatrix();

    glClear(GL_ACCUM_BUFFER_BIT);

    for (int mm = 0; mm < nLoop; mm++) {
        glClearColor(1.0, 1.0, 1.0, 0);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glPushMatrix();

        switch(nImageQuality) {
        case 1: dx = 0; dy = 0; break;
        case 2: dx = j2[mm].x; dy = j2[mm].y; break;
        case 3: dx = j3[mm].x; dy = j3[mm].y; break;
        case 4: dx = j4[mm].x; dy = j4[mm].y; break;
        case 5: dx = j8[mm].x; dy = j8[mm].y; break;
        case 6: dx = j15[mm].x; dy = j15[mm].y; break;
        case 7: dx = j24[mm].x; dy = j24[mm].y; break;
        case 8: dx = j66[mm].x; dy = j66[mm].y; break;
        default: break;
        }

        glTranslatef(dx * viewport[2] / viewport[2],
                     dy * viewport[3] / viewport[3],
                     0);


        int n_BVNum;
        int n_PVNum;
        int n_LineNum;
        int n_LinePNum;

//        if (!m_bVBO) {
            n_BVNum = ((m_nCol + 1) * m_nRow) * m_pBars[0].Polygon().size();
            n_PVNum = m_nPTNum * 3;
            n_LineNum = m_pRelations.size();
            n_LinePNum = n_LineNum * (PARTICLE + 1);

//        } else {
//            n_BVNum = ((m_nACol + 1) * m_nARow) * m_nABar;
//            n_PVNum = m_nAPVNum;
//            n_LineNum = m_nASize;
//            n_LinePNum = n_LineNum * (PARTICLE + 1);

//        }


        /**
         *initialize the VBO function
         */
        initializeGLFunctions();

        if (m_bUpdateBuffer) {
            if (glIsBuffer(m_nVBO[0])) {
                glDeleteBuffers(1, m_nVBO);
            }
        }

        if (!glIsBuffer(m_nVBO[0])) {
            glGenBuffers(1, m_nVBO);
        }

        if (m_bUpdateBuffer) {
            glBindBuffer(GL_ARRAY_BUFFER, m_nVBO[VERTICES]);

            /**
             *initialize the VBO buffer in total
             */
            glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                         + sizeof(GLfloat) * 4 * n_BVNum
                         + sizeof(GLfloat) * 2 * n_PVNum
                         + sizeof(GLfloat) * 4 * n_PVNum
                         + sizeof(GLfloat) * 2 * n_LinePNum
                         + sizeof(GLfloat) * 4 * n_LinePNum
                         + sizeof(GLfloat) * 2 * n_LinePNum
                         + sizeof(GLfloat) * 4 * n_LinePNum
                         + sizeof(GLfloat) * 2 * n_LinePNum
                         + sizeof(GLfloat) * 4 * n_LinePNum,
                         NULL, GL_STATIC_DRAW);
            /**
             * allocate the Bar's vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(GLfloat) * 2 * n_BVNum,
                            m_pBarVert);
            /**
             * allocate the Bar's color for every vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum,
                            sizeof(GLfloat) * 4 * n_BVNum, m_pBarVertColor);
            /**
             * allocate the polygon's vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum,
                            sizeof(GLfloat) * 2 * n_PVNum, m_pPolyVert);
            /**
             * allocate the polygon's color for every vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum,
                            sizeof(GLfloat) * 4 * n_PVNum, m_pPlgVertColor);

            if (m_nCluster < 5000) {
            /**
             * allocate the line's vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum
                            + sizeof(GLfloat) * 4 * n_PVNum,
                            sizeof(GLfloat) * 2 * n_LinePNum, m_pLineVert);
            /**
             * allocate the line's color for every vertex in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum
                            + sizeof(GLfloat) * 4 * n_PVNum
                            + sizeof(GLfloat) * 2 * n_LinePNum,
                            sizeof(GLfloat) * 4 * n_LinePNum, m_pLineVertColor);

            /**
             * allocate the upper boundary vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum
                            + sizeof(GLfloat) * 4 * n_PVNum
                            + sizeof(GLfloat) * 2 * n_LinePNum
                            + sizeof(GLfloat) * 4 * n_LinePNum,
                            sizeof(GLfloat) * 2 * n_LinePNum, m_pUBoundVert);

            /**
             * allocate the color of upper boundary vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum
                            + sizeof(GLfloat) * 4 * n_PVNum
                            + sizeof(GLfloat) * 2 * n_LinePNum
                            + sizeof(GLfloat) * 4 * n_LinePNum
                            + sizeof(GLfloat) * 2 * n_LinePNum,
                            sizeof(GLfloat) * 4 * n_LinePNum,
                            m_pUBoundVertColor);
            /**
             * allocate the lower boundary vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum
                            + sizeof(GLfloat) * 4 * n_PVNum
                            + sizeof(GLfloat) * 2 * n_LinePNum
                            + sizeof(GLfloat) * 4 * n_LinePNum
                            + sizeof(GLfloat) * 2 * n_LinePNum
                            + sizeof(GLfloat) * 4 * n_LinePNum,
                            sizeof(GLfloat) * 2 * n_LinePNum, m_pLBoundVert);
            /**
             * allocate the color of lower boundary vertices in VBO buffer
             */
            glBufferSubData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 2 * n_BVNum
                            + sizeof(GLfloat) * 4 * n_BVNum
                            + sizeof(GLfloat) * 2 * n_PVNum
                            + sizeof(GLfloat) * 4 * n_PVNum
                            + sizeof(GLfloat) * 2 * n_LinePNum
                            + sizeof(GLfloat) * 4 * n_LinePNum
                            + sizeof(GLfloat) * 2 * n_LinePNum
                            + sizeof(GLfloat) * 4 * n_LinePNum
                            + sizeof(GLfloat) * 2 * n_LinePNum,
                            sizeof(GLfloat) * 4 * n_LinePNum,
                            m_pLBoundVertColor);
            }
            m_bUpdateBuffer = false;
        }

        glEnableClientState(GL_VERTEX_ARRAY);
        glEnableClientState(GL_COLOR_ARRAY);

        /**
         *draw the bars
         */
        int n_BarNum;
        int n_BPNum;

//        if (!m_bVBO) {
            n_BarNum = (m_nCol + 1) * m_nRow;
            n_BPNum = m_pBars[0].Polygon().size();
//        } else {
//            n_BarNum = (m_nACol + 1) * m_nARow;
//            n_BPNum = m_nABar;
//        }

        for (int i = 0; i < n_BarNum; i++) {
            if (m_bPolyGraph || m_bLineGraph) {
                glVertexPointer(2, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                                + sizeof(GLfloat) * n_BPNum * i * 2));

                glColorPointer(4, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                               + sizeof(GLfloat) * (n_BPNum * n_BarNum * 2)
                               + sizeof(GLfloat) * n_BPNum * i * 4));

                glDrawArrays(GL_POLYGON, 0, n_BPNum);
            }
        }


        int n_PlgNum;
        int n_TriNum;

        n_PlgNum = m_pRelations.size();
        n_TriNum = m_pPolygon[0].getPolygon().size();

        /**
         * draw the lines
         */
        glLineWidth(0.01);
        for (int i = 0; i < n_PlgNum; i++) {
            if (m_pLines[i].getShow() && m_bLineGraph) {
            //if (m_bLineGraph) {
                glVertexPointer(2, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                                + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                                + sizeof(GLfloat) * n_TriNum * n_PlgNum * 4 * 3
                                + sizeof(GLfloat) *((PARTICLE + 1) * i * 2)));

                glColorPointer(4, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 4 * 3
                               + sizeof(GLfloat) * n_PlgNum * (PARTICLE + 1) * 2
                               + sizeof(GLfloat) * ((PARTICLE + 1) * i * 4)));

                glDrawArrays(GL_LINE_STRIP, 0, (PARTICLE + 1));
            }
        }


        /**
         *draw the polygons
         */
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        for (int i = 0; i < n_PlgNum; i++) {
            if (m_pPolygon[i].getShow() && m_bPolyGraph) {
            //if (m_bPolyGraph) {
                glVertexPointer(2, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                                + sizeof(GLfloat) * (n_TriNum * i * 2 * 3)));

                glColorPointer(4, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                               + sizeof(GLfloat) * (n_TriNum * i * 4 * 3)));

                glDrawArrays(GL_TRIANGLES, 0, 3 * n_TriNum);
            }
        }

        /**
         * draw the control points of each polygon
         */
//        QColor color_table[8] = {QColor(1, 0 ,0),
//                                 QColor(0, 1, 0),
//                                 QColor(0, 0, 1),
//                                 QColor(0, 1, 1),
//                                 QColor(1, 0, 1),
//                                 QColor(1, 1, 0),
//                                 QColor(0, 0, 0),
//                                 QColor(0.5, 0.5, 0)};
//        for (int i = 0; i < n_PlgNum; i++) {
//            glPointSize(10);
//            for (int p = 0; p < m_pPolygon[i].m_vCtrlPoints.size(); p++) {
//                glColor3f(color_table[p%8].red(),
//                          color_table[p%8].green(),
//                          color_table[p%8].blue());
//                glBegin(GL_POINTS);
//                glVertex2f(m_pPolygon[i].m_vCtrlPoints[p].x(),
//                           m_fHeight- m_pPolygon[i].m_vCtrlPoints[p].y());
//                glEnd();
//            }
//        }


        /**
         * draw the boundary
         */
        for (int i = 0; i < n_PlgNum; i++) {
            //glLineWidth(2.0);
            if (m_pPolygon[i].getShow() && m_bPolyGraph && m_nCluster < 50 && !m_bSubGraph) {
            //if (m_pPolygon[i].getShow() && !m_bSubGraph && m_nCluster < 5000) {
            //if (!m_bSubGraph && m_nCluster < 20) {
                glVertexPointer(2, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                                + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                                + sizeof(GLfloat) * n_TriNum * n_PlgNum * 4 * 3
                                + sizeof(GLfloat) * n_PlgNum * 2 *(PARTICLE + 1)
                                + sizeof(GLfloat) * n_PlgNum * 4 *(PARTICLE + 1)
                                + sizeof(GLfloat) * (i * 2 * (PARTICLE + 1))));

                glColorPointer(4, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 4 * 3
                               + sizeof(GLfloat) * n_PlgNum * 2 * (PARTICLE + 1)
                               + sizeof(GLfloat) * n_PlgNum * 4 * (PARTICLE + 1)
                               + sizeof(GLfloat) * n_PlgNum * 2 * (PARTICLE + 1)
                               + sizeof(GLfloat) * ((PARTICLE + 1) * i * 4)));

                glDrawArrays(GL_LINE_STRIP, 0, (PARTICLE + 1));
            }
        }

        for (int i = 0; i < n_PlgNum; i++) {
            if (m_pPolygon[i].getShow() && m_bPolyGraph && m_nCluster < 50 && !m_bSubGraph) {
            //if (m_pPolygon[i].getShow() && !m_bSubGraph && m_nCluster < 5000) {
            //if (!m_bSubGraph && m_nCluster < 20) {
                glVertexPointer(2, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                                + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                                + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                                + sizeof(GLfloat) * n_TriNum * n_PlgNum * 4 * 3
                                + sizeof(GLfloat) * n_PlgNum * 2 *(PARTICLE + 1)
                                + sizeof(GLfloat) * n_PlgNum * 4 *(PARTICLE + 1)
                                + sizeof(GLfloat) * n_PlgNum * 2 *(PARTICLE + 1)
                                + sizeof(GLfloat) * n_PlgNum * 4 *(PARTICLE + 1)
                                + sizeof(GLfloat) * (i * 2 * (PARTICLE + 1))));

                glColorPointer(4, GL_FLOAT, 0, (GLvoid *)(((char*)NULL)
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 2
                               + sizeof(GLfloat) * n_BPNum * n_BarNum * 4
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 2 * 3
                               + sizeof(GLfloat) * n_TriNum * n_PlgNum * 4 * 3
                               + sizeof(GLfloat) * n_PlgNum * 2 * (PARTICLE + 1)
                               + sizeof(GLfloat) * n_PlgNum * 4 * (PARTICLE + 1)
                               + sizeof(GLfloat) * n_PlgNum * 2 * (PARTICLE + 1)
                               + sizeof(GLfloat) * n_PlgNum * 4 * (PARTICLE + 1)
                               + sizeof(GLfloat) * n_PlgNum * 2 * (PARTICLE + 1)
                               + sizeof(GLfloat) * ((PARTICLE + 1) * i * 4)));
                glDrawArrays(GL_LINE_STRIP, 0, (PARTICLE + 1));
            }
        }

        glDisableClientState(GL_VERTEX_ARRAY);
        glDisableClientState(GL_COLOR_ARRAY);

        /*
        tagPolygon();
        drawCurveBackbone();
        textEntity();
        drawLine();
        drawBoundary();
        */
//        drawLine();
//        drawCardiograph();
        drawCurveBackbone();

        glPopMatrix();
        glAccum(GL_ACCUM, 1.0/nLoop);

    }
    glAccum(GL_RETURN, 1.0);

    glFinish();
    glDisable(GL_BLEND);

}


/**
 *draw text of entity
 */
void GLWidget::textEntity()
{
    for (int i = 0; i < m_nRow * (m_nCol + 1); i++) {
        if (m_pLabels[i].getShow()) {
            glLoadIdentity();
            qglColor(Qt::black);
            renderText(m_pLabels[i].pos().x(), m_pLabels[i].pos().y(),
                    m_pLabels[i].getLabel(), QFont("Arial", 8));
        }
    }
}

/**
 *draw the tag of relation polygon
 */
void GLWidget::tagPolygon()
{
    qglColor(Qt::black);
    float f_Width = m_fWidth * 0.75;
    float f_MovePosX_1, f_MovePosY_1, f_MovePosX_2, f_MovePosY_2;
    if (m_pMovePos.x() > f_Width) {
        f_MovePosX_1 = -m_qWidthPad;
        f_MovePosY_1 = -m_qHeightPad;
        f_MovePosX_2 = 0.0;
        f_MovePosY_2 = 0.0;
    }
    else {
        f_MovePosX_1 = 0.0;
        f_MovePosY_1 = 0.0;
        f_MovePosX_2 = m_qWidthPad;
        f_MovePosY_2 = m_qHeightPad;
    }
    renderText(m_pMovePos.x() + f_MovePosX_1 + 1, m_pMovePos.y()
               + m_qHeightPad / 2 + 3.0,
            m_sText, QFont("Arial", 10));
    glRectf(m_pMovePos.x() + f_MovePosX_1, this->height()
            - (m_pMovePos.y() - f_MovePosY_1), m_pMovePos.x() + f_MovePosX_2,
            this->height() - (m_pMovePos.y() + f_MovePosY_2));
    qglColor(Qt::white);
}


/**
 *draw the axis
 */
void GLWidget::drawAxis()
{
    for (int j = 0; j < m_nRow * (m_nCol + 1); j++) {
        qglColor(m_pBars[j].Color());
        glLoadIdentity();
        glBegin(GL_POLYGON);
        for (int i = 0; i < m_pBars[j].Polygon().size(); i++) {
            glVertex2f((*(m_pBars[j].Polygon().data() + i)).x(), this->height()
                       - (*(m_pBars[j].Polygon().data() + i)).y());
        }
        glEnd();
    }
}

/**
 *draw the polygons
 */
void GLWidget::drawPolygon()
{
    for (int j = 0; j < m_pRelations.size(); j++) {
        glLoadIdentity();
        glBegin(GL_TRIANGLES);
        for (int i = 0; i < m_pPolygon[j].getPolygon().size(); i++) {
            if (m_pPolygon[j].getShow()) {
                m_pColor = m_pPolygon[j].getColor_1();
                m_pColor.setAlphaF((m_pPolygon[j].getPolygon().size() - i * 1.0)
                                    / m_pPolygon[j].getPolygon().size());
                qglColor(m_pColor);
                glVertex2f(m_pPolygon[j].getPolygon().value(i).data()->x(),
                            this->height()
                            - m_pPolygon[j].getPolygon().value(i).data()->y());
                glVertex2f((m_pPolygon[j].getPolygon().value(i).data()
                            + 1)->x(), this->height()
                            - (m_pPolygon[j].getPolygon().value(i).data()
                            + 1)->y());
                glVertex2f((m_pPolygon[j].getPolygon().value(i).data()
                           + 2)->x(), this->height()
                           - (m_pPolygon[j].getPolygon().value(i).data()
                           + 2)->y());
            }
        }
        glEnd();
    }
}

/**
 *draw the lines
 */
void GLWidget::drawLine()
{
    QPainterPath qp_Path;
    glColor3f(0.0, 1.0, 0.0);
    if (m_bLineGraph) {
        for (int i = 0; i < m_pRelations.size(); i++) {
            if (m_pLines[i].getShow()) {
                qp_Path = m_pLines[i].getPath();
                glLoadIdentity();
                glEnable(GL_LINE_SMOOTH);
                glLineWidth(2.0f);
                for (int j = 0; j < PARTICLE; j++) {
                    qglColor(m_pPolygon[i].getColor_1());
                    qreal q_x_1 = qp_Path.pointAtPercent(j * 1.0 /
                                                         PARTICLE).x();
                    qreal q_y_1 = this->height()
                                - qp_Path.pointAtPercent(j * 1.0 /
                                                         PARTICLE).y();
                    qreal q_x_2 = qp_Path.pointAtPercent((j + 1)
                                                         * 1.0 / PARTICLE).x();
                    qreal q_y_2 = this->height()
                                - qp_Path.pointAtPercent((j + 1)
                                                         * 1.0 / PARTICLE).y();
                    glBegin(GL_LINES);
                        glVertex2f(q_x_1, q_y_1);
                        glVertex2f(q_x_2, q_y_2);
                    glEnd();
                }
            }
        }
    }
}

/**
 * @brief draw boundary for polygon
 */
void GLWidget::drawBoundary()
{
    QPainterPath qp_Path1, qp_Path2;
    glColor3f(1.0, 1.0, 1.0);
    if (m_bPolyGraph && !m_bSubGraph) {
        for (int i = 0; i < m_pRelations.size(); i++) {
            if (m_pPolygon[i].getShow()) {
                qp_Path1 = m_pPolygon[i].getPath_1();
                qp_Path2 = m_pPolygon[i].getPath_2();
                glLoadIdentity();
                glEnable(GL_LINE_SMOOTH);
                glLineWidth(0.05f);
                for (int j = 0; j < PARTICLE; j++) {
                    qreal q_x_1 = qp_Path1.pointAtPercent(j * 1.0 /
                                                          PARTICLE).x();
                    qreal q_y_1 = this->height()
                                - qp_Path1.pointAtPercent(j * 1.0 /
                                                          PARTICLE).y();
                    qreal q_x_2 = qp_Path1.pointAtPercent((j + 1) * 1.0 /
                                                          PARTICLE).x();
                    qreal q_y_2 = this->height()
                                - qp_Path1.pointAtPercent((j + 1) * 1.0 /
                                                          PARTICLE).y();
                    glBegin(GL_LINES);
                        glVertex2f(q_x_1, q_y_1);
                        glVertex2f(q_x_2, q_y_2);
                    glEnd();
                }
                for (int j = 0; j < PARTICLE; j++) {
                    qreal q_x_1 = qp_Path2.pointAtPercent(j * 1.0 /
                                                          PARTICLE).x();
                    qreal q_y_1 = this->height()
                                - qp_Path2.pointAtPercent(j * 1.0 /
                                                          PARTICLE).y();
                    qreal q_x_2 = qp_Path2.pointAtPercent((j + 1) * 1.0 /
                                                          PARTICLE).x();
                    qreal q_y_2 = this->height()
                                - qp_Path2.pointAtPercent((j + 1) * 1.0 /
                                                          PARTICLE).y();
                    glBegin(GL_LINES);
                        glVertex2f(q_x_1, q_y_1);
                        glVertex2f(q_x_2, q_y_2);
                    glEnd();
                }
            }
        }
    }
}

/**
 *draw the cardiograph backbone
 */
void GLWidget::drawCardiograph()
{
    //int n = 1000;
    //GLfloat R = 4.0f;
    //GLfloat Pi = 3.1415926536f;
    QColor color(0, 138, 0);
    qreal q_xPoint_1, q_xPoint_2;
    qreal q_yPoint_1, q_yPoint_2, q_yPoint_p;
    qreal q_Standard;
    if (m_bCardioGraph == true) {
        for (int x = 0; x < m_nCol; x++) {
            glLoadIdentity();
            qglColor(color);
            q_xPoint_1 = x * 1.0 / m_nShowedCol * m_rPanel.width()
                            + m_rPanel.left();
            q_yPoint_1 = m_fHeight - (m_fHalfPanel + m_vDiff[x] * m_qSpan);

            q_yPoint_p = m_fHalfPanel + m_vDiff[x] * m_qSpan;

            q_xPoint_2 = (x + 1) * 1.0 / m_nShowedCol * m_rPanel.width()
                            + m_rPanel.left();
            q_yPoint_2 = m_fHeight - (m_fHalfPanel + m_vDiff[x + 1] * m_qSpan);
            q_Standard = m_fHeight - (m_fHalfPanel + m_vDiff[0] * m_qSpan);

            glEnable(GL_LINE_SMOOTH);
            glLineWidth(2.0f);
            glBegin(GL_LINES);
                glVertex2f(q_xPoint_1 + 0.0f, q_Standard);
                glVertex2f(q_xPoint_1 + 5.0f, q_yPoint_1);
            glEnd();

            glBegin(GL_LINES);
                glVertex2f(q_xPoint_1 + 5.0f, q_yPoint_1);
                glVertex2f(q_xPoint_1 + 10.0f, q_yPoint_p);
            glEnd();

            glBegin(GL_LINES);
                glVertex2f(q_xPoint_1 + 10.0f, q_yPoint_p);
                glVertex2f(q_xPoint_1 + 15.0f, q_Standard);
            glEnd();

            glBegin(GL_LINES);
                glVertex2f(q_xPoint_1 + 15.0f, q_Standard);
                glVertex2f(q_xPoint_2, q_Standard);
            glEnd();
        }
        //qreal q_xLast, q_yLast;
        //q_xLast = m_nCol * 1.0 / m_nShowedCol * m_rPanel.width()
        //            + m_rPanel.left();
        //q_yLast = m_fHeight - (m_fHalfPanel + m_vDiff[m_nCol] * m_qSpan);
    }


}

/**
 *draw curve backbone
 */
void GLWidget::drawCurveBackbone()
{
    //int n = 1000;
    //GLfloat R = 4.0f;
    //GLfloat Pi = 3.1415926536f;
    QColor color(255, 0, 0);
    if (m_bCardioGraph == true) {
        for (int x = 0; x < m_nCol; x++) {
            EBundle c_Bundle;
            qreal q_xPoint_1, q_xPoint_2;
            qreal q_yPoint_1, q_yPoint_2;
            glLoadIdentity();
            qglColor(color);
            q_xPoint_1 = x * 1.0 / m_nShowedCol * m_rPanel.width()
                            + m_rPanel.left();
            q_yPoint_1 = m_fHeight -  (m_fHalfPanel + m_vDiff[x] * m_qSpan);
            q_xPoint_2 = (x + 1) * 1.0 / m_nShowedCol * m_rPanel.width()
                            + m_rPanel.left();
            q_yPoint_2 = m_fHeight - (m_fHalfPanel + m_vDiff[x + 1] * m_qSpan);

            QPointF a_1(q_xPoint_1 + (q_xPoint_2 - q_xPoint_1)/ 2.0
                        - (q_xPoint_2 - q_xPoint_1)/ 4.0, q_yPoint_1);
            QPointF b_1(q_xPoint_1 + (q_xPoint_2 - q_xPoint_1)/ 2.0
                        - (q_xPoint_2 - q_xPoint_1)/ 8.0, q_yPoint_1);
            QPointF d_1(q_xPoint_1 + (q_xPoint_2 - q_xPoint_1)/ 2.0
                        + (q_xPoint_2 - q_xPoint_1)/ 4.0, q_yPoint_1);
            QPointF m_1(q_xPoint_1 + (q_xPoint_2 - q_xPoint_1)/ 2.0,q_yPoint_1);
            QPointF p_1(q_xPoint_1, q_yPoint_1);
            QPointF p_2(q_xPoint_2, q_yPoint_2);
            QPainterPath qp_Path = c_Bundle.ctrl_Path(a_1, b_1, d_1, m_1,
                                                p_1, p_2, 0.5);
            /**
             *draw the backbone
             */
            glEnable(GL_LINE_SMOOTH);
            glLineWidth(2.0f);
            for (int i = 0; i < PARTICLE; i++) {
                qreal q_x_1 = qp_Path.pointAtPercent(i * 1.0 / PARTICLE).x();
                qreal q_y_1 = qp_Path.pointAtPercent(i * 1.0 / PARTICLE).y();
                qreal q_x_2 = qp_Path.pointAtPercent((i + 1) * 1.0 /
                                                     PARTICLE).x();
                qreal q_y_2 = qp_Path.pointAtPercent((i + 1) * 1.0/
                                                     PARTICLE).y();
                glBegin(GL_LINES);
                    glVertex2f(q_x_1, q_y_1);
                    glVertex2f(q_x_2, q_y_2);
                glEnd();
            }
            /**
             *draw the circuit
             */
            /*
            glBegin(GL_LINE_LOOP);
            for(int i = 0; i < n; i++) {
                glVertex2f(q_xPoint_1 + R * cos(2 * Pi / n * i), q_yPoint_1
                            + R * sin(2 * Pi / n * i));
            }
            glEnd();
            */
        }
        /*
        qreal q_xLast, q_yLast;
        q_xLast = m_nCol * 1.0 / m_nShowedCol * m_rPanel.width()
                    + m_rPanel.left();
        q_yLast = m_fHeight - (m_fHalfPanel + m_vDiff[m_nCol] * m_qSpan);

        glBegin(GL_LINE_LOOP);
        for(int i = 0; i < n; i++) {
            glVertex2f(q_xLast + R * cos(2 * Pi / n * i), q_yLast
                        + R * sin(2 * Pi / n * i));
        }
        glEnd();
        */
    }
}

void GLWidget::splitInterval()
{
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pLastPoint,
                                                 Qt::OddEvenFill)) {
            GLWidget::m_nNorTime = m_pPolygon[i].getTime();
            m_nTransIndex = i;
            m_bUserSelect = true;
            break;
        }
    }
//    m_pSData->splitInterval(getRelationTime());
    int n_interval = m_pSData->getCount();
    m_pGraph->write(n_interval);
    m_pHierarchy->Clear();
    m_pHierarchy->Begin(n_interval);
    m_pHierarchy->Operation(n_interval, m_nCluster);
    m_pMap->Map(n_interval);
    this->clean();
    InitGraph();
    ShowPolygons();
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
    m_nInterval = n_interval;
    m_bUpdateBuffer = true;
    repaint();
}

void GLWidget::combineInterval()
{
    m_nTime_1 = -1;
    m_nTime_2 = -1;
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pLastPoint,
                                                 Qt::OddEvenFill)) {
            m_nTime_1 = m_pPolygon[i].getTime();
            m_nTransIndex_1 = i;
        }
        if (m_pPolygon[i].getPlg().containsPoint(m_pReleasePoint,
                                                 Qt::OddEvenFill)) {
            m_nTime_2 = m_pPolygon[i].getTime();
            m_nTransIndex_2 = i;
        }
        if (m_nTime_1 != -1 && m_nTime_2 != -1) {
            break;
        }
    }
//    m_pSData->combineInterval(getRelationTime_1(), getRelationTime_2());
    int n_interval = m_pSData->getCount();
    m_pGraph->write(n_interval);
    m_pHierarchy->Clear();
    m_pHierarchy->Begin(n_interval);
    m_pHierarchy->Operation(n_interval, m_nCluster);
    m_pMap->Map(n_interval);
    this->clean();
    InitGraph();
    ShowPolygons();
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
    m_bUpdateBuffer = true;
    repaint();
}

void GLWidget::splitCluster()
{

}

void GLWidget::combineCluster()
{

}

void GLWidget::shiftOperation()
{
    //system("exec rm ../DATA/SUB/MPI_SUB");
//    for (int i = 0; i < m_pRelations.size(); i++) {
//        if (m_pPolygon[i].getPlg().containsPoint(m_pLastPoint,
//                                                 Qt::OddEvenFill)) {
//            GLWidget::m_nNorTime = m_pPolygon[i].getTime();
//            m_nTransIndex = i;
//            m_bUserSelect = true;
//            break;
//        }
//    }
    /**
     *Read the text from file
     */
    QFile file(m_sInfile1);
    QVector<QStringList> vLines;
    QVector<QStringList> vTarget;
    if (file.open(QIODevice::ReadOnly))
    {
        QTextStream in(&file);
        while (!in.atEnd())
        {
            QString str = in.readLine();
            QStringList text = str.split(QRegExp(",| "),
                                         QString::SkipEmptyParts);
            vLines.push_back(text);
        }
    }
    else
    {
        qDebug() << "Cannot open file!" << endl;
        exit(0);
    }

    for (int i = 0; i < vLines.size(); i++) {
        if (vLines[i].value(3).toStdString() == getRelationTime() &&
            vLines[i].value(0).toInt() == m_nSender &&
            vLines[i].value(1).toInt() == m_nRecver) {
            vTarget.push_back(vLines[i]);
        }
    }

    string out_s = "../DATA/SUB/MPI_SUB";
    char *outChar = const_cast<char *>(out_s.c_str());
    ofstream outFile(outChar);
    for (int i = 0; i < vTarget.size(); i++) {
        if (i != vTarget.size() - 1) {
            outFile << vTarget[i].value(0).toStdString()
                    << " "
                    << vTarget[i].value(1).toStdString()
                    << " "
                    << vTarget[i].value(2).toStdString()
                    << " "
                    << vTarget[i].value(3).toStdString()
                    << endl;
        } else {
            outFile << vTarget[i].value(0).toStdString()
                    << " "
                    << vTarget[i].value(1).toStdString()
                    << " "
                    << vTarget[i].value(2).toStdString()
                    << " "
                    << vTarget[i].value(3).toStdString();
        }
    }
    outFile.close();
}

void GLWidget::shiftOperation_full()
{
    //system("exec rm ../DATA/SUB/MPI_SUB");
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pLastPoint,
                                                 Qt::OddEvenFill)) {
            GLWidget::m_nNorTime = m_pPolygon[i].getTime();
            m_nTransIndex = i;
            m_bUserSelect = true;
            break;
        }
    }
    /**
     *Read the text from file
     */
    QFile file(m_sInfile1);
    QVector<QStringList> vLines;
    QVector<QStringList> vTarget;
    if (file.open(QIODevice::ReadOnly))
    {
        QTextStream in(&file);
        while (!in.atEnd())
        {
            QString str = in.readLine();
            QStringList text = str.split(QRegExp(",| "),
                                         QString::SkipEmptyParts);
            vLines.push_back(text);
        }
    }
    else
    {
        qDebug() << "Cannot open file!" << endl;
        exit(0);
    }

    for (int i = 0; i < vLines.size(); i++) {
        if (vLines[i].value(3).toStdString() == getRelationTime()) {
            vTarget.push_back(vLines[i]);
        }
    }

    string out_s = "../DATA/SUB/MPI_SUB";
    char *outChar = const_cast<char *>(out_s.c_str());
    ofstream outFile(outChar);
    for (int i = 0; i < vTarget.size(); i++) {
        if (i != vTarget.size() - 1) {
            outFile << vTarget[i].value(0).toStdString()
                    << " "
                    << vTarget[i].value(1).toStdString()
                    << " "
                    << vTarget[i].value(2).toStdString()
                    << " "
                    << vTarget[i].value(3).toStdString()
                    << endl;
        } else {
            outFile << vTarget[i].value(0).toStdString()
                    << " "
                    << vTarget[i].value(1).toStdString()
                    << " "
                    << vTarget[i].value(2).toStdString()
                    << " "
                    << vTarget[i].value(3).toStdString();
        }

    }
    outFile.close();
}

void GLWidget::shiftFile()
{
//    system("exec rm ../DATA/SUB/MPI_SUB");
    for (int i = 0; i < m_pRelations.size(); i++) {
        if (m_pPolygon[i].getPlg().containsPoint(m_pLastPoint,
                                                 Qt::OddEvenFill)) {
            GLWidget::m_nNorTime = m_pPolygon[i].getTime();
            m_nTransIndex = i;
            m_bUserSelect = true;
            break;
        }
    }

    stringstream ss_out;
    ss_out << GLWidget::m_nNorTime;
    string s_num = "";
    string strTREE = "../DATA/TREE/TREE_";
    if (GLWidget::m_nNorTime <= 9) {
        s_num = "0";
    }
    strTREE = strTREE + s_num + ss_out.str();
    m_pHierarchy->Operation(m_nCluster, GLWidget::m_nNorTime, m_nSubCluster,
                            strTREE,
                            m_pRelations[m_nTransIndex].Sender().toInt(),
                            m_pRelations[m_nTransIndex].Receiver().toInt());

    m_pMap->Search(getRelationTime(), m_nSender, m_nRecver, m_nNorTime);
    string strGML = "../DATA/SUB/LABEL";
    strGML = strGML;
    m_pMap->Sub(strGML);
}

/**
 *mouse move function
 */
void GLWidget::mouseMoveEvent(QMouseEvent *e)
{
    m_pMovePos = e->pos();
    UpdateTagPos();
    m_bUpdateBuffer = true;
    repaint();
}

/**
 *mouse press function
 */
void GLWidget::mousePressEvent(QMouseEvent *e)
{
    m_pLastPoint = e->pos();
    if (e->button() == Qt::LeftButton
        && e->modifiers() == Qt::ControlModifier
        && m_bXZoom)
    {
        if (m_pSData != NULL && m_pHierarchy != NULL && m_pMap != NULL) {
            splitInterval();
        }
    } else if (e->button() == Qt::LeftButton
               && e->modifiers() == Qt::ShiftModifier
               && m_bYZoom)
    {
        if (m_pSData != NULL && m_pHierarchy != NULL && m_pMap != NULL) {
            //shiftFile();
            for (int i = 0; i < m_pRelations.size(); i++) {
                if (m_pPolygon[i].getPlg().containsPoint(m_pLastPoint,
                                                         Qt::OddEvenFill)) {
                    GLWidget::m_nNorTime = m_pPolygon[i].getTime();
                    m_nTransIndex = i;
                    m_bUserSelect = true;
                    m_nSender     = m_pRelations[i].Sender().toInt();
                    m_nRecver     = m_pRelations[i].Receiver().toInt();
                    break;
                }
            }
            //cout << m_pRelations[m_nTransIndex].Sender().toStdString() << " "
                 //<< m_pRelations[m_nTransIndex].Receiver().toStdString()
                 //<< endl;
            shiftOperation();
            shiftWindow(m_pHierarchy, m_pMap, m_nNorTime, m_nCluster,
                        m_nSender, m_nRecver, getRelationTime());
        }
    } else if (e->button() == Qt::LeftButton
               && e->modifiers() == Qt::ControlModifier
               && m_bFullZoom)
    {
        shiftOperation_full();
        shiftWindow_full(m_pHierarchy, m_pMap, m_nNorTime, getRelationTime());
    } else {
        m_bMouseClick = true;
    }
}

/**
 *mouse click function
 */
void GLWidget::mouseReleaseEvent(QMouseEvent *e)
{
    if (m_bMouseClick && (e->pos() != m_pLastPoint) && m_bXZoom) {
        m_pReleasePoint = e->pos();
        combineInterval();
        m_bMouseClick = false;
    } else {
//        if (e->button() == Qt::LeftButton &&
//                e->modifiers() == Qt::ControlModifier) {
//            m_pPos = e->pos();
//            ClearPolygons();
//            UpdateSinglePolygonPos();
//        }
//        else if (e->button() == Qt::RightButton
//                 && e->modifiers() == Qt::ControlModifier) {
//            m_bClick = false;
//            ShowPolygons();
//        }
//        else if (e->button() == Qt::LeftButton &&
//                 e->modifiers() == Qt::ShiftModifier) {
//            m_pPos = e->pos();
//            if (!m_bClick) {
//                m_bClick = true;
//                ClearPolygons();
//                UpdateMSelection();
//            }
//            else {
//                UpdateMSelection();
//            }
//        }
//        else if (e->button() == Qt::RightButton
//                 && e->modifiers() == Qt::ShiftModifier) {
//            m_bClick = false;
//            ShowPolygons();
//        }
//        else {
            QWidget::mouseReleaseEvent(e);
//        }
    }
}

/**
 *mouse double click function
 */
void GLWidget::mouseDoubleClickEvent(QMouseEvent *e)
{
    if (e->button() == Qt::LeftButton) {
        m_pLDbClickPos = e->pos();
        ClearPolygons();
        ClearLines();
        UpdatePlgBar_L();
    }
    else if (e->button() == Qt::RightButton) {
        m_pRDbClickPos = e->pos();
        ClearPolygons();
        ClearLines();
        UpdatePlgBar_R();
    }
    else {
        QWidget::mouseDoubleClickEvent(e);
    }
}

/**
 *key function
 */
void GLWidget::keyPressEvent(QKeyEvent *e)
{
    qDebug() << "Response";
    if (e->key() == Qt::Key_Escape) {
        this->close();
        qDebug() << "Escape success";
    }
    else if (e->key() == '1') {
        m_nShowedCol--;
        if (m_nShowedCol < 2) {
            m_nShowedCol = 2;
        }
        qDebug() << "shrink";
        UpdateBarPos();
        UpdateLinePos();
        //UpdateLabelPos();
        UpdateBuffer();
    }
    else if (e->key() == '2') {
        m_nShowedCol++;
        if (m_nShowedCol > m_nCol) {
            m_nShowedCol = m_nCol;
        }
        qDebug() << "extend";
        UpdateBarPos();
        UpdateLinePos();
        //UpdateLabelPos();
        UpdateBuffer();
    } else {
        GLWidget::keyPressEvent(e);
    }
}

/**
 *clear all the polygons
 */
void GLWidget::ClearPolygons()
{
    for (int i = 0; i < m_pRelations.size(); i++) {
        m_pPolygon[i].setShow(false);
    }
}

/**
 *clear all the lines
 */
void GLWidget::ClearLines()
{
    for (int i = 0; i < m_pRelations.size(); i++) {
        m_pLines[i].setShow(false);
    }
}


/**
 *clear all the bars
 */
void GLWidget::ClearBars()
{
    int entity_num = m_pData->GetTotalEntitiesNum();
    int time_num = m_pData->GetTotalTimeNum();
    for (int i = 0; i < entity_num * (time_num + 1); i++) {
        m_pBars[i].setShow(false);
    }
}

/**
 *show all the polygons
 */
void GLWidget::ShowPolygons()
{
    for (int i = 0; i < m_pRelations.size(); i++) {
        m_pPolygon[i].setShow(true);
    }
}

/**
 *show all the lines
 */
void GLWidget::ShowLines()
{
    for (int i = 0; i < m_pRelations.size(); i++) {
        m_pLines[i].setShow(true);
    }
}


/**
 *show all the bars
 */
void GLWidget::ShowBars()
{
    int entity_num = m_pData->GetTotalEntitiesNum();
    int time_num = m_pData->GetTotalTimeNum();
    for (int i = 0; i < entity_num * (time_num + 1); i++) {
        m_pBars[i].setShow(true);
    }
}

/**
 * @brief GLWidget::setGraph
 */
void GLWidget::setGraph()
{
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
    m_bUpdateBuffer = true;
    repaint();
}

/**
 * @brief GLWidget::setCluster
 */
void GLWidget::setCluster()
{
    m_pData->clean();
    this->clean();
    CutCluster();
    Recompute();
}

/**
 * @brief GLWidget::setInterval
 */
void GLWidget::setInterval()
{
    m_pData->clean();
    this->clean();
    CutInterval();
    Recompute();
}

/**
 * @brief GLWidget::Recompute
 */
void GLWidget::Recompute()
{
    InitGraph();
    ShowPolygons();
    UpdateBarPos();
    UpdateLinePos();
    //UpdateLabelPos();
    UpdateBuffer();
    m_bUpdateBuffer = true;
    repaint();
}

void GLWidget::setSubCluster()
{
    m_pData->clean();
    this->clean();

    stringstream ss_out;
    ss_out << GLWidget::m_nNorTime;
    string s_num = "";
    string strTREE = "../DATA/TREE/TREE_";
    if (GLWidget::m_nNorTime <= 9) {
        s_num = "0";
    }
    strTREE = strTREE + s_num + ss_out.str();
//    strTREE.clear();
//    strTREE = "../DATA/TREE/TREE_00";
    if (!m_bfullSub) {
        m_pHierarchy->Operation(m_nC, GLWidget::m_nNorTime, m_nSubCluster,
                                strTREE, m_nSender, m_nRecver);
                              //m_pRelations[m_nTransIndex].Sender().toInt(),
                              //m_pRelations[m_nTransIndex].Receiver().toInt());
        m_pMap->Search(m_sTime, m_nSender, m_nRecver, m_nNorTime);
        string strGML = "../DATA/SUB/LABEL";
        strGML = strGML;
        m_pMap->Sub(strGML);
    } else {
        m_pHierarchy->Operation(GLWidget::m_nNorTime, m_nSubCluster, strTREE);
        m_pMap->Search(m_sTime);
        string strGML = "../DATA/SUB/LABEL";
        strGML = strGML;
        m_pMap->Sub(strGML);
    }
    Recompute();
}

void GLWidget::setSubAxis(int gradient)
{

    if (m_bSubSet == true) {
        m_bSubSet = false;
        return;
    }
    m_nSubCluster = gradient;
}

/**
 *Size impact axis
 */
void GLWidget::setXAxis(int gradient)
{
    if (m_bXSet == true) {
        m_bXSet = false;
        return;
    }
    m_fThickness = gradient * 1.0;

//    for (int i = 0; i <= m_nCol; i++) {
//        qDebug() << "The maximum in the column "
//                 << i
//                 << " is "
//                 << m_pData->getColMax(i);
//    }
}

/**
 *Cluster difference axis
 */
void GLWidget::setYAxis(int gradient)
{
    if (m_bYSet == true) {
        m_bYSet = false;
        return;
    }
//    m_qSpan = gradient;
    m_qSpan = gradient * 0.1;
//    m_qSpan = gradient * 0.01;
}

/**
 *Time interval axis
 */
void GLWidget::setZAxis(int gradient)
{
    if (m_bZSet == true) {
        m_bZSet = false;
        return;
    }
    m_nInterval = gradient;
}

/**
 *Cluster number axis
 */
void GLWidget::setKAxis(int gradient)
{
    if (m_bKSet == true) {
        m_bKSet = false;
        return;
    }
    m_nCluster = gradient;
}

/**
 *Column scale axia
 */
void GLWidget::setSAxis(int gradient)
{
    if (m_bSSet == true) {
        m_bSSet = false;
        return;
    }
    m_nShowedCol = gradient;
}

/**
 * @brief Cardiograph checkbox
 * @param check: 0, 1, 2
 */
void GLWidget::setCState(int check)
{
    if (check == 0) {
        m_bCardioGraph = false;
        repaint();
        return;
    }
    if (check == 1) {
        return;
    }
    if (check == 2) {
        m_bCardioGraph = true;
        repaint();
        return;
    }
}

/**
 * @brief Polygon show
 * @param check: 0, 1, 2
 */
void GLWidget::setKState(int check)
{
    if (check == 0) {
        m_bPolyGraph = false;
        ClearPolygons();
        repaint();
        return;
    }
    if (check == 1) {
        return;
    }
    if (check == 2) {
        m_bPolyGraph = true;
        ShowPolygons();
        repaint();
        return;
    }
}

/**
 * @brief Line show
 * @param check: 0, 1, 2
 */
void GLWidget::setLState(int check)
{
    if (check == 0) {
        m_bLineGraph = false;
        ClearLines();
        repaint();
        return;
    }
    if (check == 1) {
        return;
    }
    if (check == 2) {
        m_bLineGraph = true;
        ShowLines();
        repaint();
        return;
    }
}

/**
 * @brief GLWidget::setXZoomState
 * @param check: 0, 1, 2
 */
void GLWidget::setXZoomState(int check)
{
    if (check == 0) {
        m_bXZoom = false;
        m_pSData->zoomFlush();
        return;
    }
    if (check == 1) {
        return;
    }
    if (check == 2) {
        m_bXZoom = true;
        m_pSData->zoomInitialize(m_nInterval);
        return;
    }
}

/**
 * @brief GLWidget::setYZoomState
 * @param check: 0, 1, 2
 */
void GLWidget::setYZoomState(int check)
{
    if (check == 0) {
        m_bYZoom = false;
        return;
    }
    if (check == 1) {
        return;
    }
    if (check == 2) {
        m_bYZoom = true;
        return;
    }
}

/**
 * @brief GLWidget::setYZoomState
 * @param check: 0, 1, 2
 */
void GLWidget::setFullZoomState(int check)
{
    if (check == 0) {
        m_bFullZoom = false;
        return;
    }
    if (check == 1) {
        return;
    }
    if (check == 2) {
        m_bFullZoom = true;
        return;
    }
}

/**
 * @brief GLWidget::Init
 */
void GLWidget::Init()
{
    int n_interval = m_nInterval;
    m_pSData = new SData();
    m_pSData->Begin(n_interval);
    n_interval = m_pSData->getCount();
    m_pGraph = new WriteGraph();
    m_pGraph->write(n_interval);
    m_pHierarchy = new Hierarchy();
    m_pHierarchy->Begin(n_interval);
    m_pHierarchy->Operation(n_interval, m_nCluster);
//    return;
    m_pMap = new CMap();
    m_pMap->Map(n_interval);
    m_bXSet = true;
    m_bYSet = true;
    m_bZSet = true;
    m_bKSet = true;
    m_bSSet = true;
    m_bSubSet = true;
}

/**
 * @brief GLWidget::CutInterval
 */
void GLWidget::CutInterval()
{
    int n_interval = m_nInterval;
    m_pSData->Operation(n_interval);
    n_interval = m_pSData->getCount();
    m_pGraph->write(n_interval);
    m_pHierarchy->Clear();
    m_pHierarchy->Begin(n_interval);
    m_pHierarchy->Operation(n_interval, m_nCluster);
    m_pMap->Map(n_interval);
    m_nShowedCol = m_nInterval;
    emit sAxisChanged(m_nInterval);
}

/**
 * @brief GLWidget::CutCluster
 */
void GLWidget::CutCluster()
{
    int n_interval = m_pSData->getCount();
    m_pHierarchy->Operation(n_interval, m_nCluster);
    m_pMap->Map(n_interval);
}

/**
 * @brief GLWidget::InitGraph
 */
void GLWidget::InitGraph()
{
    /**
     *initailize the data
     */
    m_pData = new CData();
    m_pData->getFileName(m_sInfile1);

    m_pGroups = new Cluster(m_pData->GetEntities());

    int n_fileNum = m_pData->GetTimes().size();
    for (int i = 0; i < n_fileNum; i++) {
        m_pGroups->setCP(i);
    }

    m_pData->EntityPos(m_pGroups->getPos());
    m_pData->EntityColor(m_pGroups->getColors());

    m_nENum = m_pData->GetTotalEntitiesNum();

    /**
     *get the cluster difference from the file
     */
    if (m_sInfile2 != NULL) {
        m_vDiff.clear();
        m_pGroups->setPathPos(m_sInfile2);
        qreal qr_tmp = 0.0;
        qreal qr_mtp = 1.0;
        for(int i = 0; i < m_pGroups->getDiff().size(); i++) {
            m_vDiff << qr_tmp + qr_mtp * (m_pGroups->getDiff().value(i) * 1.0f);
        }
        m_vDiff << 0;
    }

    /**
     *initalize the bars
     */
    int entity_num   = m_pData->GetTotalEntitiesNum();
    int time_num     = m_pData->GetTotalTimeNum();
    float max_value1 = m_pData->GlbMaxAmount();
    float max_value  = m_pData->MaxAmount();
//    m_fGlbMax = max_value1 > m_fGlbMax ? max_value1 : m_fGlbMax;
//    float glbMax     = 5.39257e9;
    float glbMax     = max_value1;
//    float glbMax       = max_value;
//    float glbMax       = 8388608;
//    float glbMax       = 16777216;
//    float glbMax       = 134217728;
//    float glbMax       = 67108864;
//    qDebug() << glbMax;

    m_pBars = new CBar[entity_num * (time_num + 1)];
    for (int t = 0; t <= time_num; t++) {
        for (int e = 0; e < entity_num; e++) {
            int id = e + t * entity_num;
            m_pBars[id].EntityIndex(id);
            m_pBars[id].TimeIndex(t);
            m_pBars[id].SendValue(m_pData->SendAmount(e, t) / max_value);
            m_pBars[id].RecvValue(m_pData->RecvAmount(e, t) / max_value);
             m_pBars[id].Color(m_pGroups->getColor((id % m_nENum) % 2));
//            m_pBars[id].Color(m_pData->SendAmount(e, t) / max_value);
//            m_pBars[id].Color(m_pData->EntityColor(id));
        }
    }

    /**
     *set the position for lines and initialize the lines
     */
    m_pRelations = m_pData->GetRelations();
    m_pLines = new Line[m_pRelations.size()];
    m_pPolygon = new CPolygon[m_pRelations.size()];
    for (int i = 0; i < m_pRelations.size(); i++)
    {
        m_pPolygon[i].setShow(false);
        /**
         *colored according to the message size
         */
        //if (!m_bSubGraph) {
            m_pPolygon[i].setColor_1(m_pGroups->getColor(
                                         m_pRelations[i].Amount() / glbMax));
//                                     m_pRelations[i].Amount() / max_value1));
            m_pPolygon[i].setColor_2(m_pGroups->getColor(
                                         m_pRelations[i].Amount() / glbMax));
//                                     m_pRelations[i].Amount() / max_value1));
            m_pLines[i].Color_1(m_pGroups->getColor(
                                    m_pRelations[i].Amount() / glbMax));
//                                    m_pRelations[i].Amount() / max_value1));
            m_pLines[i].Color_2(m_pGroups->getColor(
                                    m_pRelations[i].Amount() / glbMax));
//                                    m_pRelations[i].Amount() / max_value1));
        //}

        /**
         *colored according to the position
         */
        //else {
            //m_pPolygon[i].setColor_1(m_pData->EntityColor(m_pRelations[i]
            //                                              .SenderIndex()));
            //m_pPolygon[i].setColor_2(m_pData->EntityColor(m_pRelations[i]
            //                                             .ReceiverIndex()));

            //m_pLines[i].Color_1(m_pData->EntityColor(m_pRelations[i]
            //                                         .SenderIndex()));
            //m_pLines[i].Color_2(m_pData->EntityColor(m_pRelations[i]
            //                                         .ReceiverIndex()));
        //}
        m_pLines[i].Lines(i);
        m_pBars[m_pRelations[i].SenderIndex()].pushSend(i);
        m_pBars[m_pRelations[i].ReceiverIndex()].pushRecv(i);
    }

    /**
     *initialize the label
     */
    m_pLabels = new CLabel[entity_num * (time_num + 1)];
    for (int t = 0; t <= time_num; t++) {
        for (int e = 0; e < entity_num; e++) {
            int id = e + t * entity_num;
            m_pLabels[id].EntityIndex(id);
            m_pLabels[id].TimeIndex(t);
            m_pLabels[id].setLabel(m_pData->GetEntity(id));
        }
    }

    /**
     *set the arguement for the whole graph
     */
    int n_max = 0;

    /**
     *adjustable parameter
     */
    m_nCol       = time_num;
    m_nRow       = entity_num;
    for (int x = 0; x < m_nCol; x++) {
        if (x == m_nCol) {
            x = x - 1;
        }
        n_max = m_pData->getColMax(x) > m_pData->getColMax(n_max) ?
                    x : n_max;
    }
    m_nMax= n_max;
    adjSize(m_nMax);
}

/**
 * @brief GLWidget::clean
 */
void GLWidget::clean()
{
    if (m_pBars != NULL) {
        delete[] m_pBars;
        m_pBars = NULL;
    }
    if (m_pLabels != NULL) {
        delete[] m_pLabels;
        m_pLabels = NULL;
    }
    if (m_pLines != NULL) {
        delete[] m_pLines;
        m_pLines = NULL;
    }
    if (m_pPolygon != NULL) {
        delete[] m_pPolygon;
        m_pPolygon = NULL;
    }
    if (m_pGroups != NULL) {
        delete m_pGroups;
    }
}

/**
 * @brief GLWidget::adjSize
 * @param n_max
 */
void GLWidget::adjSize(int n_max)
{
//    m_fFixedSize = (m_fHeight - 2 * m_fPad) / (m_nRow) + m_fThickness *
//                                                           (m_pData->getColMax(n_max)
//                                                            / m_pData->MaxAmount());
    m_fFixedSize = (m_fHeight - 2 * m_fPad) / (m_nRow + m_fThickness *
                                               (m_pData->getColMax(n_max)
                                                / m_pData->MaxAmount()));
    m_fAdjSize   = m_fThickness * m_fFixedSize;
    m_fHalfPanel = m_fHeight / 2;
    m_bClick     = false;
}

void GLWidget::save_buffer(ofstream &outf)
{
    GLfloat x;
    GLfloat y;
    GLfloat red;
    GLfloat green;
    GLfloat blue;
    GLfloat alphaF;

    int             n_BVNum;
    int             n_PVNum;
    int             n_Col;
    int             n_Row;
    int             n_Bar;
    int             n_Plg;
    int             n_Size;
    int             n_LineSize;

    //int             n_ShowedCol;
    //float           f_Span;
    //QVector<float>   v_Diff;

    n_Col       = m_nCol;
    n_Row       = m_nRow;
    n_Bar       = m_pBars[0].Polygon().size();
    n_Plg       = m_pPolygon[0].getPolygon().size();
    n_Size      = m_pRelations.size();
    n_LineSize  = m_pRelations.size() * (PARTICLE + 1);

    //n_ShowedCol = m_nShowedCol;
    //f_Span      = m_qSpan;
    //v_Diff      = m_vDiff;


    n_BVNum     = ((m_nCol + 1) * m_nRow) * m_pBars[0].Polygon().size();
    n_PVNum     = m_nPTNum * 3;

    outf.write(reinterpret_cast<char *>(&n_BVNum), sizeof(int));
    outf.write(reinterpret_cast<char *>(&n_PVNum), sizeof(int));
    outf.write(reinterpret_cast<char *>(&n_Col), sizeof(int));
    outf.write(reinterpret_cast<char *>(&n_Row), sizeof(int));
    outf.write(reinterpret_cast<char *>(&n_Bar), sizeof(int));
    outf.write(reinterpret_cast<char *>(&n_Plg), sizeof(int));
    outf.write(reinterpret_cast<char *>(&n_Size), sizeof(int));


    //outf.write(reinterpret_cast<char *>(&n_Plg), sizeof(int));
    //outf.write(reinterpret_cast<char *>(&n_Size), sizeof(int));
    //for (int i = 0; i < v_Diff.size(); i++) {
    //  outf.write(reinterpret_cast<char *>(&v_Diff[i]), sizeof(int));
    //}


    assert(m_pBarVert != NULL);
    assert(m_pBarVertColor != NULL);
    assert(m_pPolyVert != NULL);
    assert(m_pPlgVertColor != NULL);
    assert(m_pLineVert != NULL);
    assert(m_pLineVertColor != NULL);
    assert(m_pUBoundVert != NULL);
    assert(m_pUBoundVertColor != NULL);
    assert(m_pLBoundVert != NULL);
    assert(m_pLBoundVertColor != NULL);

    for (int i = 0; i < n_BVNum; i++) {
        x = m_pBarVert[i].x;
        y = m_pBarVert[i].y;
        red = m_pBarVertColor[i].red;
        green = m_pBarVertColor[i].green;
        blue = m_pBarVertColor[i].blue;
        alphaF = m_pBarVertColor[i].alphaF;
        outf.write(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
    }

    for (int i = 0; i < n_PVNum; i++) {
        x = m_pPolyVert[i].x;
        y = m_pPolyVert[i].y;
        red = m_pPlgVertColor[i].red;
        green = m_pPlgVertColor[i].green;
        blue = m_pPlgVertColor[i].blue;
        alphaF = m_pPlgVertColor[i].alphaF;
        outf.write(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
    }

    for (int i = 0; i < n_LineSize; i++) {
        x = m_pLineVert[i].x;
        y = m_pLineVert[i].y;
        red = m_pLineVertColor[i].red;
        green = m_pLineVertColor[i].green;
        blue = m_pLineVertColor[i].blue;
        alphaF = m_pLineVertColor[i].alphaF;
        outf.write(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
    }

    for (int i = 0; i < n_LineSize; i++) {
        x = m_pUBoundVert[i].x;
        y = m_pUBoundVert[i].y;
        red = m_pUBoundVertColor[i].red;
        green = m_pUBoundVertColor[i].green;
        blue = m_pUBoundVertColor[i].blue;
        alphaF = m_pUBoundVertColor[i].alphaF;
        outf.write(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
    }

    for (int i = 0; i < n_LineSize; i++) {
        x = m_pLBoundVert[i].x;
        y = m_pLBoundVert[i].y;
        red = m_pLBoundVertColor[i].red;
        green = m_pLBoundVertColor[i].green;
        blue = m_pLBoundVertColor[i].blue;
        alphaF = m_pLBoundVertColor[i].alphaF;
        outf.write(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        outf.write(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
    }
}

void GLWidget::read_buffer(ifstream &inf)
{
    GLfloat x;
    GLfloat y;
    GLfloat red;
    GLfloat green;
    GLfloat blue;
    GLfloat alphaF;

    int n_BVNum;
    int n_PVNum;
    int n_Col;
    int n_Row;
    int n_Bar;
    int n_Plg;
    int n_Size;
    int n_LineSize;

//    int             n_ShowedCol;
//    float           f_Span;
//    QVector<float>   v_Diff;

    inf.read(reinterpret_cast<char *>(&n_BVNum), sizeof(int));
    inf.read(reinterpret_cast<char *>(&n_PVNum), sizeof(int));
    inf.read(reinterpret_cast<char *>(&n_Col), sizeof(int));
    inf.read(reinterpret_cast<char *>(&n_Row), sizeof(int));
    inf.read(reinterpret_cast<char *>(&n_Bar), sizeof(int));
    inf.read(reinterpret_cast<char *>(&n_Plg), sizeof(int));
    inf.read(reinterpret_cast<char *>(&n_Size), sizeof(int));

    m_nACol     = n_Col;
    m_nARow     = n_Row;
    m_nABar     = n_Bar;
    m_nAPlg     = n_Plg;
    m_nASize    = n_Size;
    m_nAPVNum   = n_PVNum;
    n_LineSize  = m_nASize * (PARTICLE + 1);

    if (m_pBarVert != NULL) {
        delete [] m_pBarVert;
    }
    if (m_pBarVertColor != NULL) {
        delete [] m_pBarVertColor;
    }
    if (m_pPolyVert != NULL) {
        delete [] m_pPolyVert;
    }
    if (m_pPlgVertColor != NULL) {
        delete [] m_pPlgVertColor;
    }

    m_pBarVert          = new CVert[n_BVNum];
    m_pBarVertColor     = new CVColor[n_BVNum];
    m_pPolyVert         = new CVert[n_PVNum];
    m_pPlgVertColor     = new CVColor[n_PVNum];
    m_pLineVert         = new CVert[n_LineSize];
    m_pLineVertColor    = new CVColor[n_LineSize];
    m_pUBoundVert       = new CVert[n_LineSize];
    m_pUBoundVertColor  = new CVColor[n_LineSize];
    m_pLBoundVert       = new CVert[n_LineSize];
    m_pLBoundVertColor  = new CVColor[n_LineSize];

    for (int i = 0; i < n_BVNum; i++) {
        inf.read(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
        m_pBarVert[i].x = x;
        m_pBarVert[i].y = y;
        m_pBarVertColor[i].red = red;
        m_pBarVertColor[i].green = green;
        m_pBarVertColor[i].blue = blue;
        m_pBarVertColor[i].alphaF = alphaF;
    }

    for (int i = 0; i < n_PVNum; i++) {
        inf.read(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
        m_pPolyVert[i].x = x;
        m_pPolyVert[i].y = y;
        m_pPlgVertColor[i].red = red;
        m_pPlgVertColor[i].green = green;
        m_pPlgVertColor[i].blue = blue;
        m_pPlgVertColor[i].alphaF = alphaF;
    }

    for (int i = 0; i < n_LineSize; i++) {
        inf.read(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
        m_pLineVert[i].x = x;
        m_pLineVert[i].y = y;
        m_pLineVertColor[i].red = red;
        m_pLineVertColor[i].green = green;
        m_pLineVertColor[i].blue = blue;
        m_pLineVertColor[i].alphaF = alphaF;
    }

    for (int i = 0; i < n_LineSize; i++) {
        inf.read(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
        m_pUBoundVert[i].x = x;
        m_pUBoundVert[i].y = y;
        m_pUBoundVertColor[i].red = red;
        m_pUBoundVertColor[i].green = green;
        m_pUBoundVertColor[i].blue = blue;
        m_pUBoundVertColor[i].alphaF = alphaF;
    }

    for (int i = 0; i < n_LineSize; i++) {
        inf.read(reinterpret_cast<char *>(&x), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&y), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&red), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&green), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&blue), sizeof(GLfloat));
        inf.read(reinterpret_cast<char *>(&alphaF), sizeof(GLfloat));
        m_pLBoundVert[i].x = x;
        m_pLBoundVert[i].y = y;
        m_pLBoundVertColor[i].red = red;
        m_pLBoundVertColor[i].green = green;
        m_pLBoundVertColor[i].blue = blue;
        m_pLBoundVertColor[i].alphaF = alphaF;
    }

}
