/****************************************************************************
*
*   Copyright (c) 2006 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*   $Id: schedule.h 2917 2011-03-29 15:30:17Z carrick $
****************************************************************************/
#if !defined( SCHEDULE_H )
#define SCHEDULE_H

#include "types.h"

/**
 * Maximum number of possible events
 **/
#define SCHEDULE_MAX_POSSIBLE_EVENTS 30

/**
 * Maximum number of sec events that can be scheduled
 **/
#define SCHEDULE_MAX_SEC_EVENTS 15

/**
 * Maximum number of ms events that can be scheduled
 **/
#define SCHEDULE_MAX_MS_EVENTS 15

/**
 * If events are missed because scheduleProcessSec() was not called on
 * every second then when it is called events from the previous
 * SCHEDULE_RUN_WINDOW_SECS will be run as well if they have not
 * already been run.  This allows events to be run even when some
 * events run longer than a second.
 **/
#define SCHEDULE_RUN_WINDOW_SECS 10

/**
 * If events are missed because scheduleProcessMS() was not called on
 * every ms then when it is called events from the previous
 * SCHEDULE_RUN_WINDOW_MS will be run as well if they have not
 * already been run.  This allows events to be run even when some
 * events run longer than a ms.
 **/
#define SCHEDULE_RUN_WINDOW_MS 120


/**
 * Adds an event to make it possible to schedule it from the console.
 * Returns 0 on success, else - error.
 *
 * @param callback function to run
 * @param descript ion of the function
 * @return 0 on success, else - error.
 **/
eint8 scheduleAddPossibleEvent(void (*callback)(), char *descript);

/**
 * Processes scheduled events that take place with second granularity.
 * sec indicates the current time in seconds, although it doesn't need
 * to correspond to the real world time (just so long as subsequent
 * calls are monitonically increasing).  If seconds are skipped then
 * events from previous times will be processed up to some window of
 * time.  If sec is less than or equal to the lastRun time then we don't run.
 *
 * N.B. Be careful that sec does not roll over!
 *
 *
 * @param sec the time in seconds
 * @return the number of events that were processed
 **/
euint32 scheduleProcessSec(unsigned int sec);

/**
 * Processes scheduled events that take place with millisecond
 * granularity.  ms indicates the current time in ms, although it
 * doesn't need to correspond to the real world time (just so long as
 * subsequent calls are monitonically increasing).  If ms are skipped
 * then events from previous times will be processed up to some window
 * of time.  If ms is less than or equal to the lastRun time then we don't run.
 *
 * N.B. Be careful that ms does not roll over!
 *
 * @param ms the time in milliseconds
 * @return the number of events that were processed
 **/
euint32 scheduleProcessMS(euint32 ms);

/**
 *  Schedule an event to run every sec seconds.  If sec is 0 then it
 *  is not run. Returns an ID unique to this event or -1 on error.
 *
 * @param sec interval in seconds between runs
 * @param callback a void function to be called every sec seconds
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 * @return event id or -1 on error
 **/
eint32 scheduleSec(unsigned int sec, void (*callback)(), char *descript, int priority);

/**
 *  Schedule an event to run every ms milliseconds.  If ms is 0 then
 *  it is not run. Returns an ID unique to this event or -1 on error.
 *
 * @param ms interval in milliseconds between runs
 * @param callback a void function to be called every ms
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 * @return event id or -1 on error
 **/
int scheduleMS(euint32 ms, void (*callback)(), char *descript, int priority);

/**
 * Updates event id to run every sec seconds instead of its old rate.
 * Returns 1 iff successfully updated, else 0.
 *
 * @param id of the process to update
 * @param sec new run interval in seconds
 * @return 1 iff success, else 0
 **/
int scheduleSecUpdateInterval(int id, unsigned int sec);

/**
 * Updates event id to run every ms milliseconds instead of its old rate.
 * Returns 1 iff successfully updated, else 0.
 *
 * @param id of the process to update
 * @param ms new run interval in ms
 * @return 1 iff success, else 0
 **/
int scheduleMSUpdateInterval(int id, euint32 ms);

/**
 * Flags an event based on id to be deleted next time the scheduler is
 * run.  The event is deleted before it is run next time round.
 * Returns 1 if deleted, else 0.
 **/
int scheduleDeleteID(int id);

/**
 * Flags an event based on handler to be deleted next time the
 * scheduler is run.  The event is deleted before it is run next time
 * round.  Returns 1 if deleted, else 0.
 **/
int scheduleDeleteHandler(void (*callback)());

/**
 * Flags an event as to be deleted next time the scheduler is run
 * based on the string description.  The event is deleted before it is
 * run next time round.  Returns 1 if deleted, else 0.
 **/
int scheduleDeleteString(char *str);

/**
 *  Schedule an event to run in sec seconds from now.  Once it has run
 *  it is removed from the run queue.  Returns an ID unique to this
 *  event or -1 on error.
 *
 * @param sec time from now at which event will be run
 * @param callback a void function to be called at the specified time
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 **/
int scheduleRunOnceSec(unsigned int sec, void (*callback)(), char *descript, int priority);

/**
 *  Schedule an event to run in ms milliseconds from now.  Once it has run
 *  it is removed from the run queue.  Returns an ID unique to this
 *  event or -1 on error.
 *
 * @param ms time from now at which event will be run
 * @param callback a void function to be called at the specified time
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 **/
int scheduleRunOnceMS(euint32 ms, void (*callback)(), char *descript, int priority);

/**
 * Returns 1 if we have already been init'ed else 0.
 **/
eint8 scheduleIsInitialized(void);

/**
 * Initializes the scheduler.  Returns 0 on success, else - error.  Ok
 * to call multiple times.
 **/
eint8 scheduleInit(void);


/****************************************************************
 ********** Probably don't need to use these ********************
 ****************************************************************/

/**
 * All info about a scheduled command.
 **/
typedef struct {
  void (*handler)();  ///<the handler
  int id;
  ///If once==1 then interval is the next time to run and it will only run once
  int once;
  int delete; ///<1 if this should be deleted next time round, else 0
  int priority; ///<0 is lowest priority, 10 is highest
  char *description; ///<description
  euint32 interval; ///<Interval to run at if > 0
} scheduled_events_t;

/**
 * All info about a possible
 **/
typedef struct {
  void (*handler)();  ///<the handler
  char *description; ///<description
} possible_events_t;

/**
 * All of the events that are scheduled
 **/
extern scheduled_events_t scheduledSecEvents[SCHEDULE_MAX_SEC_EVENTS];

/**
 * The number of currently scheduled events
 **/
extern volatile int numScheduledSecEvents;

/**
 * All of the events that are scheduled
 **/
extern scheduled_events_t scheduledMSEvents[SCHEDULE_MAX_MS_EVENTS];

/**
 * The number of currently scheduled milli second events
 **/
extern volatile int numScheduledMSEvents;

/**
 * All possible events.
 **/
extern possible_events_t possibleEvents[SCHEDULE_MAX_POSSIBLE_EVENTS];

/**
 * The number of possibleEvents.
 **/
extern int numPossibleEvents;


#endif /* SCHEDULE_H */
