/****************************************************************************
*
*   Copyright (c) 2006 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*   $Id: schedule.c 2917 2011-03-29 15:30:17Z carrick $
****************************************************************************/

#include "schedule.h"
#include "types.h"
#include <string.h>

/**
 * 1 iff the scheduler system has been inited, else 0.
 **/
eint8 scheduleInitialized = 0;

/**
 * All of the events that are scheduled
 **/
scheduled_events_t scheduledSecEvents[SCHEDULE_MAX_SEC_EVENTS];

/**
 * The number of currently scheduled events
 **/
volatile int numScheduledSecEvents = 0;

/**
 * All of the events that are scheduled
 **/
scheduled_events_t scheduledMSEvents[SCHEDULE_MAX_MS_EVENTS];

/**
 * The number of currently scheduled milli second events
 **/
volatile int numScheduledMSEvents = 0;

/**
 * All possible events.
 **/
possible_events_t possibleEvents[SCHEDULE_MAX_POSSIBLE_EVENTS];

/**
 * The number of possibleEvents.
 **/
int numPossibleEvents = 0;

/**
 * Last time that events were processed (in seconds).
 **/
euint32 lastRunSecs = 0;

/**
 * Last time that events were processed (in ms).
 **/
euint32 lastRunMS = 0;

//some protos
static int scheduleGetFreeID(void);
int scheduleDeleteReally(int id);

/**
 * Processes scheduled events that take place with tick granularity.
 * tick indicates the current time in ticks, although it doesn't need
 * to correspond to the real world time (just so long as subsequent
 * calls are monitonically increasing).  If ticks are skipped then
 * events from previous times will be processed up to runLastWindow
 * time.  If ticks is less than or equal to lastRun then we don't run.
 *
 * N.B. Be careful that tick does not roll over!
 *
 * @param tick current tick number
 * @param lastRun last run tick
 * @param runLastWindow number of ticks to process old events which haven't been processed
 * @param numScheduledEvents number of scheduled events
 * @param scheduledEvents the scheduled events
 * @return the number of events that were processed
 **/
static int scheduleProcess(euint32 tick, euint32 *lastRun,
                           int runLastWindow, volatile int *numScheduledEvents,
                           scheduled_events_t *scheduledEvents){
  int numRun = 0;
  int j;
  euint32 i;
  int nomoredeletes=0;

  //Make sure we haven't already processed this tick.
  if(tick <= *lastRun) return 0;

  //Delete events which should be deleted
  while(nomoredeletes == 0){
    nomoredeletes = 1;
    for(j=0;j<*numScheduledEvents;j++){
      if(scheduledEvents[j].delete == 1){
        scheduleDeleteReally(scheduledEvents[j].id);
        nomoredeletes = 0;
        break;
      }
    }
  }

  //Proccess old events if they were missed up to our specified window
  if(tick - (*lastRun) > runLastWindow){
    (*lastRun) = tick - runLastWindow;
  }

  //Loop over all events
  for(j=0;j<*numScheduledEvents;j++){
    //Run once events
    if((scheduledEvents[j].once == 1) && (scheduledEvents[j].interval < tick)){
      if(scheduledEvents[j].delete == 0){
        scheduledEvents[j].handler();
        scheduledEvents[j].delete = 1;
        numRun++;
      }
      continue;
    }

    //Loop over all times since the last run
    for(i=(*lastRun)+1;i<=tick;i++){
      //See if it should run
      if((scheduledEvents[j].interval >= 0)
         && (i%(scheduledEvents[j].interval) == 0)
         && scheduledEvents[j].once == 0){
        scheduledEvents[j].handler();
        numRun++;

        break;
      }
    }
  }

  (*lastRun) = tick;
  return numRun;
}



/**
 * Processes scheduled events that take place with second granularity.
 * sec indicates the current time in seconds, although it doesn't need
 * to correspond to the real world time (just so long as subsequent
 * calls are monitonically increasing).  If seconds are skipped then
 * events from previous times will be processed up to some window of
 * time.  If sec is less than or equal to lastRun time then we don't run.
 *
 * N.B. Be careful that sec does not roll over!
 *
 * @param sec the time in seconds
 * @return the number of events that were processed
 **/
euint32 scheduleProcessSec(unsigned int sec){
  return scheduleProcess((euint32)sec,&lastRunSecs,SCHEDULE_RUN_WINDOW_SECS,
                         &numScheduledSecEvents,scheduledSecEvents);

}

/**
 * Processes scheduled events that take place with millisecond
 * granularity.  ms indicates the current time in ms, although it
 * doesn't need to correspond to the real world time (just so long as
 * subsequent calls are monitonically increasing).  If ms are skipped
 * then events from previous times will be processed up to some window
 * of time.  If ms is less than or equal to the lastRun time then we don't run.
 *
 * N.B. Be careful that ms does not roll over!
 *
 * @param ms the time in milliseconds
 * @return the number of events that were processed
 **/
euint32 scheduleProcessMS(euint32 ms){
  return scheduleProcess(ms,&lastRunMS,SCHEDULE_RUN_WINDOW_MS,
                         &numScheduledMSEvents,scheduledMSEvents);
}

/**
 * Adds an event to make it possible to schedule it from the console.
 * Returns 0 on success, else - error.
 *
 * @param callback function to run
 * @param descript ion of the function
 * @return 0 on success, else - error.
 **/
eint8 scheduleAddPossibleEvent(void (*callback)(), char *descript){
  //Check bounds
  if(numPossibleEvents >= SCHEDULE_MAX_POSSIBLE_EVENTS){
    return -1;
  }

  possibleEvents[numPossibleEvents].handler = callback;
  possibleEvents[numPossibleEvents].description = descript;

  numPossibleEvents++;

  return 0;
}

/**
 *  Schedule an event to run every sec seconds.  If sec is 0 then it
 *  is not run. Returns an ID unique to this event or -1 on error.
 *
 * @param sec interval in seconds between runs
 * @param callback a void function to be called every sec seconds
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 * @return event id or -1 on error
 **/
eint32 scheduleSec(unsigned int sec, void (*callback)(), char *descript, int priority){
  //See if we have room to add it
  if(numScheduledSecEvents >= SCHEDULE_MAX_SEC_EVENTS){
    return -1;
  }

  scheduledSecEvents[numScheduledSecEvents].handler = callback;
  scheduledSecEvents[numScheduledSecEvents].interval = (euint32) sec;
  scheduledSecEvents[numScheduledSecEvents].id = scheduleGetFreeID();
  scheduledSecEvents[numScheduledSecEvents].description = descript;
  scheduledSecEvents[numScheduledSecEvents].once = 0;
  scheduledSecEvents[numScheduledSecEvents].delete = 0;
  scheduledSecEvents[numScheduledSecEvents].priority = priority;

  numScheduledSecEvents++;

  return scheduledSecEvents[numScheduledSecEvents-1].id;
}

/**
 *  Schedule an event to run every ms milliseconds.  If ms is 0 then
 *  it is not run. Returns an ID unique to this event or -1 on error.
 *
 * @param ms interval in milliseconds between runs
 * @param callback a void function to be called every ms
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 * @return event id or -1 on error
 **/
int scheduleMS(euint32 ms, void (*callback)(), char *descript, int priority){
  //See if we have room to add it
  if(numScheduledMSEvents >= SCHEDULE_MAX_MS_EVENTS){
    return -1;
  }

  scheduledMSEvents[numScheduledMSEvents].handler = callback;
  scheduledMSEvents[numScheduledMSEvents].interval = ms;
  scheduledMSEvents[numScheduledMSEvents].id = scheduleGetFreeID();
  scheduledMSEvents[numScheduledMSEvents].description = descript;
  scheduledMSEvents[numScheduledMSEvents].once = 0;
  scheduledMSEvents[numScheduledMSEvents].delete = 0;
  scheduledMSEvents[numScheduledMSEvents].priority = priority;

  numScheduledMSEvents++;

  return scheduledMSEvents[numScheduledMSEvents-1].id;
}

/**
 * Updates event id to run every interval instead of its old rate.
 * Event is the array of events to look in which contains numevents.
 * Returns 1 iff successfully updated, else 0.
 *
 * @param id of the process to update
 * @param sec new run interval in seconds
 * @param event array of events
 * @param numevents in event
 * @return 1 iff success, else 0
 **/
static int scheduleUpdateInterval(int id, euint32 interval, scheduled_events_t *event, int numevents){
  int i;

  //Loop over all events
  for(i=0;i<numevents;i++){
    //See if the id matches
    if(event[i].id == id){
      event[i].interval = interval;
      return 1;
    }
  }

  return 0;
}

/**
 * Updates event id to run every sec seconds instead of its old rate.
 * Returns 1 iff successfully updated, else 0.
 *
 * @param id of the process to update
 * @param sec new run interval in seconds
 * @return 1 iff success, else 0
 **/
int scheduleSecUpdateInterval(int id, unsigned int sec){
  return scheduleUpdateInterval(id,sec,scheduledSecEvents,numScheduledSecEvents);
}

/**
 * Updates event id to run every ms milliseconds instead of its old rate.
 * Returns 1 iff successfully updated, else 0.
 *
 * @param id of the process to update
 * @param ms new run interval in ms
 * @return 1 iff success, else 0
 **/
int scheduleMSUpdateInterval(int id, euint32 ms){
  return scheduleUpdateInterval(id,ms,scheduledMSEvents,numScheduledMSEvents);
}

/**
 * Delete event identified by id from event which contains numevents.
 * Returns 1 iff sucessfully deleted, else 0.
 **/
static int scheduleDeleteFrom(int id,scheduled_events_t *event, volatile int *numevents){
  int i,j;

  //Loop over all events
  for(i=0;i<(*numevents);i++){
    //See if the id matches
    if(event[i].id == id){
      //Set the fram to zero
      //Move everything in the tail forward
      for(j=i;j<(*numevents)-1;j++){
        event[j].handler = event[j+1].handler;
        event[j].interval = event[j+1].interval;
        event[j].id = event[j+1].id;
        event[j].description = event[j+1].description;
        event[j].once = event[j+1].once;
        event[j].delete = event[j+1].delete;
        event[j].priority = event[j+1].priority;
      }
      //Decrease the number of events
      (*numevents)--;
      return 1;
    }
  }

  return 0;
}

/**
 * Really delete event identified by id.  Returns 1 iff sucessfully deleted, else 0.
 **/
int scheduleDeleteReally(int id){
  int ret = 0;

  ret = scheduleDeleteFrom(id,scheduledSecEvents,&numScheduledSecEvents);
  ret |= scheduleDeleteFrom(id,scheduledMSEvents,&numScheduledMSEvents);

  return ret;
}

/**
 * Flags an event based on id to be deleted next time the scheduler is
 * run.  The event is deleted before it is run next time
 * round. Returns 1 if deleted, else 0.
 **/
int scheduleDeleteID(int id){
  int i;
  int ret = 0;

  for(i=0;i<numScheduledMSEvents;i++){
    if(scheduledMSEvents[i].id == id){
      scheduledMSEvents[i].delete = 1;
      ret = 1;
    }
  }
  for(i=0;i<numScheduledSecEvents;i++){
    if(scheduledSecEvents[i].id == id){
      scheduledSecEvents[i].delete = 1;
      ret = 1;
    }
  }

  return ret;
}

/**
 * Flags an event based on handler to be deleted next time the
 * scheduler is run.  The event is deleted before it is run next time
 * round. Returns 1 if deleted, else 0.
 **/
int scheduleDeleteHandler(void (*callback)()){
  int i;
  int ret = 0;

  for(i=0;i<numScheduledMSEvents;i++){
    if((scheduledMSEvents[i].handler)
       == ( callback)){
      scheduledMSEvents[i].delete = 1;
      ret = 1;
    }
  }
  for(i=0;i<numScheduledSecEvents;i++){
    if((scheduledSecEvents[i].handler)
       == ( callback)){
      scheduledSecEvents[i].delete = 1;
      ret = 1;
    }
  }

  return ret;
}

/**
 * Flags an event as to be deleted next time the scheduler is run
 * based on the string description.  The event is deleted before it is
 * run next time round. Returns 1 if deleted, else 0.
 **/
int scheduleDeleteString(char *str){
  int i;
  int ret = 0;

  for(i=0;i<numScheduledMSEvents;i++){
    if(strncmp(scheduledMSEvents[i].description,str,100) == 0){
      scheduledMSEvents[i].delete = 1;
      ret = 1;
    }
  }
  for(i=0;i<numScheduledSecEvents;i++){
    if(strncmp(scheduledSecEvents[i].description,str,100) == 0){
      scheduledSecEvents[i].delete = 1;
      ret = 1;
    }
  }

  return ret;
}

/**
 *  Schedule an event to run in sec seconds from now.  Once it has run
 *  it is removed from the run queue.  Returns an ID unique to this
 *  event or -1 on error.
 *
 * @param sec time from now at which event will be run
 * @param callback a void function to be called at the specified time
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 **/
int scheduleRunOnceSec(unsigned int sec, void (*callback)(), char *descript, int priority){
  //See if we have room to add it
  if(numScheduledSecEvents >= SCHEDULE_MAX_SEC_EVENTS){
    return -1;
  }

  scheduledSecEvents[numScheduledSecEvents].handler = callback;
  scheduledSecEvents[numScheduledSecEvents].interval = lastRunSecs + sec;
  scheduledSecEvents[numScheduledSecEvents].id = scheduleGetFreeID();
  scheduledSecEvents[numScheduledSecEvents].description = descript;
  scheduledSecEvents[numScheduledSecEvents].once = 1;
  scheduledSecEvents[numScheduledSecEvents].delete = 0;
  scheduledSecEvents[numScheduledSecEvents].priority = priority;

  numScheduledSecEvents++;

  return scheduledSecEvents[numScheduledSecEvents-1].id;
}

/**
 *  Schedule an event to run in ms milliseconds from now.  Once it has run
 *  it is removed from the run queue.  Returns an ID unique to this
 *  event or -1 on error.
 *
 * @param ms time from now at which event will be run
 * @param callback a void function to be called at the specified time
 * @param descript a short description of the event
 * @param priority level of this event 0 to 10, where 0 is low priority, 10 is high
 **/
int scheduleRunOnceMS(euint32 ms, void (*callback)(), char *descript, int priority){
  //See if we have room to add it
  if(numScheduledMSEvents >= SCHEDULE_MAX_MS_EVENTS){
    return -1;
  }

  scheduledMSEvents[numScheduledMSEvents].handler = callback;
  scheduledMSEvents[numScheduledMSEvents].interval = lastRunMS + ms;
  scheduledMSEvents[numScheduledMSEvents].id = scheduleGetFreeID();
  scheduledMSEvents[numScheduledMSEvents].description = descript;
  scheduledMSEvents[numScheduledMSEvents].once = 1;
  scheduledMSEvents[numScheduledMSEvents].delete = 0;
  scheduledMSEvents[numScheduledMSEvents].priority = priority;

  numScheduledMSEvents++;

  return scheduledMSEvents[numScheduledMSEvents-1].id;
}

/**
 * Gets the next free process id.
 **/
static int scheduleGetFreeID(void){
  int i;
  char used[SCHEDULE_MAX_MS_EVENTS+SCHEDULE_MAX_SEC_EVENTS+1];

  //Init
  for(i=0;i<SCHEDULE_MAX_MS_EVENTS+SCHEDULE_MAX_SEC_EVENTS+1;i++){
    used[i] = 0;
  }

  //Fill in any used
  for(i=0;i<numScheduledMSEvents;i++){
    used[scheduledMSEvents[i].id] = 1;
  }
  for(i=0;i<numScheduledSecEvents;i++){
    used[scheduledSecEvents[i].id] = 1;
  }

  //Now find a free one
  for(i=0;i<SCHEDULE_MAX_MS_EVENTS+SCHEDULE_MAX_SEC_EVENTS+1;i++){
    if(used[i] == 0) break;
  }

  return i;
}


/**
 * Returns 1 if we have already been init'ed else 0.
 **/
eint8 scheduleIsInitialized(void){
  return scheduleInitialized;
}


/**
 * Initializes the scheduler.  Returns 0 on success, else - error.  Ok
 * to call multiple times.
 **/
eint8 scheduleInit(void){
  if(scheduleInitialized == 1) return 0;
  scheduleInitialized = 1;

  return 0;
}

