/****************************************************************************
*
*   Copyright (c) 2006 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
*   02110-1301 USA
*
*   $Id: console_custom.h 2917 2011-03-29 15:30:17Z carrick $
****************************************************************************/


#if !defined( CONSOLE_CUSTOM_H )
#define CONSOLE_CUSTOM_H

#include <inttypes.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <avr/pgmspace.h>
#include "types.h"

/**********************************************************************
 * Below are macros which _MUST_ be defined
 **********************************************************************/

/**
 * Add PSS wrapper to consoleSend to make sure it stores strings in
 * program memory space.
 **/
#define consoleSend(str, ... ) __consoleSend__(PSS( str ), ##__VA_ARGS__ )

/**
 * The maximum length of the user defined prompt.
 **/
#define NAME_SIZE 16

/**
 * The maximum size of the send buffer.  Only needs to be the width
 * the longest line. Should be less than the capacity of UINT8_T.
 **/
#define BUFTX_LEN 100

/**
 * The maximum size of the receive buffer.  Only needs to be the width
 * the longest line. Should be less than the capacity of UINT8_T.
 **/
#define BUFRX_LEN 100

/**
 * The maximum number of commands that we can have.  Must be less than
 * the capacity of UINT8_T.
 **/
#define MAX_COMMANDS 32

/**
 * Different types that are needed.  Must at least meet the minimum
 * specified number of bits (eg uint8_t is an unsigned 8-bit type).
 **/
#define INT8_T int8_t
#define UINT8_T uint8_t
#define INT16_T int16_t
#define UINT16_T uint16_t
#define INT32_T int32_t
#define UINT32_T uint32_t
#define CHAR_T char
#define UCHAR_T unsigned char

/**********************************************************************
 * Below are macros which are optional.  Comment out if not used.
 **********************************************************************/

/**
 * A default name for the console prompt.  
 **/
#ifndef NAME
#define NAME "Hoverboard"
#endif


/**
 * Commands to be executed before consoleSend(...)
 **/
//#define CONSOLE_SEND_PRE()

/**
 * Commands to be executed at the end of consoleSend(...)
 **/
#define CONSOLE_SEND_POST() {consoleSendBufferUsed = 0;}

/**
 * If this is define then the rx and tx buffer share the same address
 * space and the size of the Rx buffer is used for both.
 **/
//#define USE_SAME_RXTX_BUFFER

/**********************************************************************
 * User defined functions below.
 **********************************************************************/

void handleHelp(void);

/**
 * Used to convert a Program Space String for use with sprintf or the
 * such.  The string stored in program space is copied into
 * consoleSendBuffer before being sent.  consoleSend will clear
 * consoleSendBuffer after it is done.  Use PSS("str") to
 * transparently use a string in sprintf.
 **/
char *pssToStr(const char *s);

extern volatile uint8_t consoleSendBufferUsed;

/**
 * Used to create a Program Space String for use with sprintf or the
 * such.  This string is stored in program space but is copied into
 * consoleSendBuffer before being used.  sendBuffer will clear
 * consoleSendBuffer after it is done. 
 *
 * Usage:
 *  snprintf(buf,BUF_LEN,PSS("Print a %s from program space."),PSS("string"));
 *  sendBuffer();
 **/
#define PSS(str) (pssToStr(PSTR((str))))

#endif /* CONSOLE_CUSTOM_H */
