/****************************************************************************
*
*   Copyright (c) 2008 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   Some parts taken from Marsette Vona's cmb/robostix/main.c
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*   $Id: console_custom.c 2917 2011-03-29 15:30:17Z carrick $
****************************************************************************/

#include "console.h"
#include "uart.h"
//Interrupt control
#include <avr/interrupt.h>
//Watchdog timer
#include <avr/wdt.h>


#define CONSOLE_SEND_BUFFER_SIZE 100

/**
 * <p>1 if consoleInit() has been called, else 0.</p>
 **/
INT8_T consoleInitialized = 0;

/**
 * How much of consoleSendBuffer is currently used.
 **/
volatile uint8_t consoleSendBufferUsed = 0;

/**
 * Buffer for using program space strings with consoleSendStr.
 **/
char consoleSendBuffer[CONSOLE_SEND_BUFFER_SIZE];

/**
 * Used to convert a Program Space String for use with sprintf or the
 * such.  The string stored in program space is copied into
 * consoleSendBuffer before being sent.  consoleSend will clear
 * consoleSendBuffer after it is done.  Use PSS("str") to
 * transparently use a string in sprintf.
 **/
char *pssToStr(const char *s){
  char c;
  char *start = consoleSendBuffer+consoleSendBufferUsed;
  for(c = pgm_read_byte(s); c; ++s, c = pgm_read_byte(s)){
    if(consoleSendBufferUsed >= CONSOLE_SEND_BUFFER_SIZE-1){
      //Make sure it is in bounds and not really high
      consoleSendBufferUsed = CONSOLE_SEND_BUFFER_SIZE-1;
      break;
    }
    consoleSendBuffer[consoleSendBufferUsed++] = c;
  }
  consoleSendBuffer[consoleSendBufferUsed++] = 0;

  return start;
}

/*****************************************************************
 * Functions which _must_ be filled in.
 *****************************************************************/

/**
 * Receives a byte.  Non-blocking, returns 1 iff byte received.
 **/
UINT8_T consoleReceiveByte(CHAR_T *c){
  if(uartReceiveByte(1, c) == 1) return 1;
  return uartReceiveByte(0, c);
}

/**
 * Sends a byte on the console.
 **/
void consoleSendByte(CHAR_T c){
  uartSendByte(1,(unsigned char) c);
  uartSendByte(0,(unsigned char) c);
}

/**
 * Sends cnt bytes on the console.
 **/
void consoleSendBytes(CHAR_T *c, UINT8_T cnt){
  UINT8_T i;
  for(i=0;i<cnt;i++){
    consoleSendByte(c[i]);
  }
}

/*****************************************************************
 * Functions which might be useful to modify
 *****************************************************************/

/**
 * <p>Returns 1 if the console is already initialized, else 0.</p>
 **/
INT8_T consoleIsInitialized(){
  return consoleInitialized;
}

/**
 * Reboots the board into the bootloader immediately.
 **/
void handleReboot(void){
 //The way we reboot is to start the watch dog timer with a very short
 //timeout.  The watchdog will reboot the board if it is not fed
 //within the specified time.  After we start it, we just wait to get
 //rebooted.  We could alternatively just jump to the start of the
 //bootloader in flash, but this will change with different
 //processors.  We disable interrupts so that noone can reset the WDT
 //in an interrupt.
  cli(); //Disable interrupts
  wdt_enable(WDTO_15MS);
  while(1);
}

void handleReadMem(void){
  char *tok;
  uint16_t adr;
  uint8_t val;
  if(!getToken(&tok)) return;
  sscanf(tok,"%x",&adr);

  val = *(uint8_t *)adr;
  consoleSend("Reading memory address %#x\r\n",adr);
  consoleSend("Value: %#hhx\r\n",val);
}

void handleWriteMem(void){
  char *tok;
  uint16_t adr;
  uint8_t val;
  if(!getToken(&tok)) return;
  sscanf(tok,"%x",&adr);
  if(!getToken(&tok)) return;
  sscanf(tok,"%hhx",&val);
  consoleSend("Writing memory address %#x to %#hhx\r\n",adr,val);
  *(uint8_t *)adr = val;

}


/**
 * Performs any initialization which is necessary for the console to
 * start.
 **/
void consoleInit(void){
  //If we have already been called, return
  if(consoleInitialized == 1) return;
  consoleInitialized = 1;

  //This adds a command on its own
  consoleAddCommand(handleHelp,
                    "help",
                    "h",
                    "",
                    "this");
  consoleAddCommand(handleReboot,
                    "reboot",
                    "",
                    "",
                    "Reboot the board now");


  //This creates a "menu" that contains subcommands
  int idx =
  consoleAddSubCommand(NULL,
                       "memory",
                       "",
                       "",
                       "Functions raw mem access",
                       CONSOLE_MENU,
                       0);
  //Add items to the "menu"
  consoleAddSubCommand(handleReadMem,
                       "readmem",
                       "",
                       "<addr(hex)>",
                       "Read memory at the specified location",
                       CONSOLE_COMMAND,
                       idx);
  consoleAddSubCommand(handleWriteMem,
                       "writemem",
                       "",
                       "<addr(hex) val(hex)>",
                       "Write memory val at the specified addr",
                       CONSOLE_COMMAND,
                       idx);





  //Set it up as a vt100 term
  consoleSend(VT100_INIT);
  //consoleSend(VT100_CLEAR);

  //handleHelp();
  consoleSend("\r\n");

  snprintf(boxName,NAME_SIZE,NAME);

  printPrompt();
};

cmd_info_t cmd_help;
/**
 * Prints the help message
 **/
void handleHelp(void){
  
  cmd_help.name = "Help";
  cmd_help.descr = "This";

  consoleSend("# cmds: %d\r\n", num_cmds);

  consolePrintMenu(&cmd_help,0);
}
