/****************************************************************************
*
*   Copyright (c) 2006 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*
*   $Id: console.h 2917 2011-03-29 15:30:17Z carrick $
****************************************************************************/


#if !defined( CONSOLE_H )
#define CONSOLE_H

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "console_custom.h"

#ifdef __cplusplus
extern "C" {
#endif 

#define DEL 0x7f
#define BS 0x08

#define VT100_INIT "\x1B\x63"
#define VT100_CLEAR "\x1B[2J"
#define VT100_CURSOR_VISIBLE "\x1B[?25h"
#define VT100_CURSOR_HIDDEN "\x1B[?25l"
#define VT100_BS "\x1B[1D"

#define VT100_CLEARFONT  "\x1B[0m"
#define VT100_RESET      "\x1B[0m"
#define VT100_BOLD       "\x1B[1m"
#define VT100_DIM        "\x1B[2m"
#define VT100_UNDERLINE  "\x1B[4m"
#define VT100_BLINK      "\x1B[5m"
#define VT100_REVERSE    "\x1B[7m"
#define VT100_HIDDEN     "\x1B[8m"

#define  VT100_BLACK    "\x1B[30m"
#define  VT100_RED      "\x1B[31m"
#define  VT100_GREEN    "\x1B[32m"
#define  VT100_YELLOW   "\x1B[33m"
#define  VT100_BLUE     "\x1B[34m"
#define  VT100_MAGENTA  "\x1B[35m"
#define  VT100_CYAN     "\x1B[36m"
#define  VT100_WHITE    "\x1B[37m"

#define VT100_ON_BLACK    "\x1B[40m"
#define VT100_ON_RED      "\x1B[41m"
#define VT100_ON_GREEN    "\x1B[42m"
#define VT100_ON_YELLOW   "\x1B[43m"
#define VT100_ON_BLUE     "\x1B[44m"
#define VT100_ON_MAGENTA  "\x1B[45m"
#define VT100_ON_CYAN     "\x1B[46m"
#define VT100_ON_WHITE    "\x1B[47m"

//first param is line, second is col
#define VT100_SET_CURSOR_POS "\x1B[%d;%dH"

/**
 * Pointer to a function taking no parameters and returning nothing.
 **/
  typedef void (*handler_t)();

/**
 * Used with {@link consoleAddSubCommand()} to specify that this is
 * actually a menu to which sub commands can be added
 **/
#define CONSOLE_MENU 1

/**
 * Used with {@link consoleAddSubCommand()} to specify taht this is
 * actually a command to be executed.
 **/
#define CONSOLE_COMMAND 0

/**
 * Make sure there is some default name
 **/
#ifndef NAME
#define NAME "Console"
#endif  

/**
 * All info about a command.
 **/
typedef struct {
  handler_t handler;  ///<the handler
  CHAR_T *name; ///<command name
  CHAR_T *shortname; ///<short cmd name, "" for none
  CHAR_T *param_help; ///<params description

  CHAR_T *descr; ///<description
  UINT8_T type; ///<type of cmd CONSOLE_MENU or CONSOLE_COMMAND
  UINT8_T parent; ///<parent of this node
} cmd_info_t;

/**
 * Where all of the commands are stored
 **/
extern cmd_info_t cmd_info[];

/**
 * The current number of commands
 **/
extern UINT8_T num_cmds;

/**
 * Name of this box
 **/
extern CHAR_T boxName[NAME_SIZE];


/**
 * Add a command to the console.  The type is either CONSOLE_MENU to
 * indicate that this is a menu or CONSOLE_COMMAND to indicate this is
 * a command to be executed.  When you add a command or menu specify
 * parent 0 to indicate that it should be displayed in the main help
 * message.  Otherwise, specify the id returned when creating a parent
 * to add it as a sub command.  Note that command names and shortcuts
 * must be unique.  The handler can be NULL for menus, but most not be
 * for other commands.
 *
 * @param handler void function handle
 * @param name of the command used to call this fuction from console
 * @param shortname of the command (an abbreviation) ignored for menus
 * @param param_help description of the parameters (normally in <params>)
 * @param descr of the function for help message display
 * @param type of command CONSOLE_MENU or CONSOLE_COMMAND
 * @param parent of this command (of type CONSOLE_MENU)
 * @return -1 on error and >=0 id used for CONSOLE_MENU types to add
 *           further commands
 **/
INT32_T consoleAddSubCommand(handler_t handler, CHAR_T *name,
                              CHAR_T *shortname, CHAR_T *param_help,
                              CHAR_T *descr,
                              UINT8_T type,
                              UINT16_T parent);
/**
 * Add a command to the console.  Calls {@link consoleAddSubCommand()}
 * with type CONSOLE_COMMAND and parent 0.
 *
 * @return 1 iff successfully added
 **/
UINT8_T consoleAddCommand(handler_t handler, CHAR_T *name,
                           CHAR_T *shortname, CHAR_T *param_help,
                           CHAR_T *descr);

/**
 * Prints the prompt.
 **/
void printPrompt(void);

/**
 * Prints a message saying the command was unknown.
 **/
void unknownCommand(void);

/**
 * Prints the menu for the specified command.  The id is the id of
 * this node which children use to specify this node as the parent.
 * This impl really only uses cmd for the command name and description
 * string.
 **/
  void consolePrintMenu(cmd_info_t *cmd, int id);

/**
 * Gets the next token, returns 0 if there are no more.  If the send
 * and receive buffer are the same then this should be called before
 * any sends are done because it uses the same buffer.
 **/
UINT8_T getToken(CHAR_T **token);

/**
 * Version of console send using a va_list instead of a variable
 * number of arguments.
 **/
void consoleVSend(CHAR_T *str, va_list argp);

/**
 * Sends something over the console using printf type syntax.
 **/
void __consoleSend__(CHAR_T *str, ...);

/**
 * If consoleSend has not been previously defined, just define it as
 * the default function.
 **/
#ifndef consoleSend
#define consoleSend(str, ... ) __consoleSend__(str, ##__VA_ARGS__ )
#endif

/**
 * Sends a single byte from the console.
 **/
void consoleSendByte(CHAR_T c);

/**
 * Sends cnt bytes on the console.
 **/
void consoleSendBytes(CHAR_T *c, UINT8_T cnt);

/**
 * Function which tries to receive a single byte on the console.
 **/
UINT8_T consoleReceiveByte(CHAR_T *c);

/**
 * Callback to be called anytime a byte is available.
 **/
void consoleProcess(void);

/**
 * <p>Returns 1 if the console is already initialized, else 0.</p>
 **/
INT8_T consoleIsInitialized();

/**
 * Init the console program.  Ok to be called multiple times.
 **/
void consoleInit(void);

#ifdef __cplusplus
}
#endif 

#endif /* CONSOLE_H */
