/****************************************************************************
*
*   Copyright (c) 2006 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   Some parts originally taken from Marsette Vona's cmb/robostix/main.c
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*   $Id: console.c 2917 2011-03-29 15:30:17Z carrick $
****************************************************************************/

#include "console.h"

#ifndef USE_SAME_RXTX_BUFFER
#define BUFTX_LEN_REAL BUFTX_LEN
#define BUFRX_LEN_REAL BUFRX_LEN
CHAR_T bufTx[BUFTX_LEN_REAL];
CHAR_T bufRx[BUFRX_LEN_REAL];
#else
#define BUFTX_LEN_REAL BUFRX_LEN
#define BUFRX_LEN_REAL BUFRX_LEN
CHAR_T bufRx[BUFRX_LEN_REAL];
CHAR_T *bufTx = bufRx;
#endif


/**
 * The name of this box
 **/
CHAR_T boxName[NAME_SIZE];

/**
 * The current number of commands;
 **/
UINT8_T num_cmds = 0;

/**
 * These are were all of the commands are defined.  The fields are:
 **/
cmd_info_t cmd_info[MAX_COMMANDS];

/**
 * Add a command to the console.  The type is either CONSOLE_MENU to
 * indicate that this is a menu or CONSOLE_COMMAND to indicate this is
 * a command to be executed.  When you add a command or menu specify
 * parent 0 to indicate that it should be displayed in the main help
 * message.  Otherwise, specify the id returned when creating a parent
 * to add it as a sub command.  Note that command names and shortcuts
 * must be unique.  The handler can be NULL for menus, but most not be
 * for other commands.
 *
 * @param handler void function handle
 * @param name of the command used to call this fuction from console
 * @param shortname of the command (an abbreviation) ignored for menus
 * @param param_help description of the parameters (normally in <params>)
 * @param descr of the function for help message display
 * @param type of command CONSOLE_MENU or CONSOLE_COMMAND
 * @param parent of this command (of type CONSOLE_MENU)
 * @return -1 on error and >=0 id used for CONSOLE_MENU types to add
 *           further commands
 **/
INT32_T consoleAddSubCommand(handler_t handler, CHAR_T *name,
                              CHAR_T *shortname, CHAR_T *param_help,
                              CHAR_T *descr,
                              UINT8_T type,
                              UINT16_T parent){
  if(num_cmds >= MAX_COMMANDS)
    return 0;

  cmd_info[num_cmds].handler = handler;
  cmd_info[num_cmds].name = name;
  cmd_info[num_cmds].shortname = shortname;
  cmd_info[num_cmds].param_help = param_help;
  cmd_info[num_cmds].descr = descr;
  cmd_info[num_cmds].type = type;
  cmd_info[num_cmds].parent = parent;

  num_cmds++;
  return num_cmds-1;
}

/**
 * Add a command to the console.  Calls {@link consoleAddSubCommand()}
 * with type CONSOLE_COMMAND and parent 0.
 *
 * @return 1 iff successfully added
 **/
UINT8_T consoleAddCommand(handler_t handler, CHAR_T *name,
                           CHAR_T *shortname, CHAR_T *param_help,
                           CHAR_T *descr){
  if(consoleAddSubCommand(handler,name,shortname,param_help,descr,
                          CONSOLE_COMMAND,0) >= 0){
    return 1;
  }else{
    return 0;
  }
}


/**
 * Version of console send using a va_list instead of a variable
 * number of arguments.
 **/
void consoleVSend(CHAR_T *str, va_list argp){
  UINT8_T i;

  //Put it in our tx buffer
  vsnprintf(bufTx,BUFTX_LEN_REAL-1,str,argp);  

  //Now send everything
  for (i = 0; i < BUFTX_LEN_REAL; i++) {
    if (bufTx[i] == '\0'){
      break;
    }
    //consoleSendByte(bufTx[i]);
  }
  consoleSendBytes(bufTx,i);
}

/**
 * Sends something over the console using printf type syntax.
 **/
void __consoleSend__(CHAR_T *str, ...){
  va_list argp;

#ifdef CONSOLE_SEND_PRE
  CONSOLE_SEND_PRE();
#endif 

  va_start(argp,str);
  //vsnprintf(bufTx,5,str,argp);
  consoleVSend(str,argp);
  va_end(argp);


#ifdef CONSOLE_SEND_POST
  CONSOLE_SEND_POST();
#endif 
}


/**
 * Prints the prompt
 **/
void printPrompt(void){
  //Show the prompt
  consoleSend("%s> ", boxName); 
}

/**
 * Prints a message saying the command was unknown.
 **/
void unknownCommand(void){
  consoleSend("\r\nUnknown cmd, h help.\r\n");
}

/**
 * Number of tokens left to process
 **/
UINT8_T numTokens = 0;

/**
 * The current position of getToken
 **/
UINT8_T getTokenPos = 0;

/**
 * Gets the next token, returns 0 if there are no more.  This should
 * be called before any sends are done because it uses the same
 * buffer.
 **/
UINT8_T getToken(CHAR_T **token){
  
  if(numTokens <= 0)
    return 0;

  //Find the first non-null
  while(bufRx[getTokenPos]== '\0'){
    getTokenPos++;
    if(--numTokens <= 0)
      return 0;
  }

  numTokens--;
  *token = bufRx+getTokenPos;

  //Now go to the end of the string
  while(bufRx[getTokenPos++] != '\0' && getTokenPos<BUFRX_LEN_REAL);

  return 1;
}

/**
 * The possition in buf that addCHAR_TToTokens is currently using for
 * processing.
 **/
UINT16_T addChar_ToTokensPos = 0;

/**
 * Adds a CHAR_T to the current set of tokens (tokens are white space
 * deliminated). Returns 1 if a new line has been received (indicating
 * end of the command) or of the buffer has been filled.
 **/
static UINT8_T addChar_ToTokens(CHAR_T c) {
  //If we are starting over, reset numTokens
  if(addChar_ToTokensPos == 0){
    numTokens = 0;
    getTokenPos = 0;
  }

  if(addChar_ToTokensPos < BUFRX_LEN_REAL) {
    //also echo a newline when we get a carriage return
    if (c == '\r')
      consoleSendByte('\n');

    //echo back
    if(!(c == BS || c == DEL)) 
      consoleSendByte(c);

    //end of string?
    if ((c == '\0') 
        || (c == '\r') 
        || (c == '\n')){
      //terminate string token
      bufRx[addChar_ToTokensPos] = '\0';
      addChar_ToTokensPos = 0;
      numTokens++;
      return 1;
    }

    //We are moving on to a new token (unless it is escaped)
    if (c == ' '){
      //See prev char was a backspace
      if(addChar_ToTokensPos > 0 && bufRx[addChar_ToTokensPos-1] == '\\'){
        //If it was replace it with the space (done below)
        addChar_ToTokensPos--;
        //bufRx[addChar_ToTokensPos] = ' ';
        //return 0;
      }else{
        numTokens++;
        bufRx[addChar_ToTokensPos++] = '\0';
        return 0;
      }
    }

    //handle backspace
    if ((c == BS) || (c == DEL)){
      UINT8_T j;
      if (addChar_ToTokensPos > 0){
        addChar_ToTokensPos--;

        //Keep the number of tokens consistent 
        if(bufRx[addChar_ToTokensPos] == '\0'){
          numTokens--;
        }

        //do it this way for maximum portability
        for (j = 0; j < 4; j++)
          consoleSendByte(VT100_BS[j]);
        consoleSendByte(' ');
        for (j = 0; j < 4; j++)
          consoleSendByte(VT100_BS[j]);
      }
    } else {
      bufRx[addChar_ToTokensPos++] = c;
    }
  }else{
    //Overflowed our buffer, terminate and return
    bufRx[BUFRX_LEN_REAL-1] = '\0';
    addChar_ToTokensPos = 0;
    return 1;
  }
  return 0;
}

/**
 * Prints the menu for the specified command.  The id is the id of
 * this node which children use to specify this node as the parent.
 * This impl really only uses cmd for the command name and description
 * string.
 **/
void consolePrintMenu(cmd_info_t *cmd, int id){
#define TMP_BUF_LEN 40
  CHAR_T tmpBuff[TMP_BUF_LEN];
  int i;
  consoleSend("%s Help for command: %s\r\n",boxName, cmd->name);
  consoleSend("                     %s\r\n",cmd->descr);
  consoleSend("Commands followed by '*' are items which have subitems\r\n");
  consoleSend("These subitems can be accessed 'help subitem' or just 'subitem'\r\n");

  consoleSend("   Command,short      <params>               Descript\r\n");
  
  for (i = 0; i < num_cmds; i++) {
    if(cmd_info[i].parent == id){
      snprintf(tmpBuff,TMP_BUF_LEN,"%s",cmd_info[i].name);
      if(strcmp(cmd_info[i].shortname,"")!=0){
        snprintf(tmpBuff+strlen(tmpBuff),TMP_BUF_LEN,",%s",cmd_info[i].shortname);
      }
      if(cmd_info[i].type == CONSOLE_MENU){
        snprintf(tmpBuff+strlen(tmpBuff),TMP_BUF_LEN,"*");
      }
      consoleSend("   %-18s %-22s %s.\r\n",
                  tmpBuff,cmd_info[i].param_help, cmd_info[i].descr);
    }
  }
  
}

void consoleProcess(void){
  CHAR_T c;
  //Process any bytes we have
  while(consoleReceiveByte(&c)){

    //Proccess it and see if we have a full command
    if(addChar_ToTokens(c)){
      UINT8_T i;
      CHAR_T *cmd;

      //Try to get the next token
      if(!getToken(&cmd)){
        printPrompt(); 
        return;
      }

      for (i = 0; i < num_cmds; i++) {
        if (strcmp(cmd_info[i].name, cmd) == 0
            || (strcmp(cmd_info[i].shortname, cmd) == 0
                &&(strcmp(cmd_info[i].shortname,"") != 0))){
          if(cmd_info[i].type == CONSOLE_MENU){
            consolePrintMenu(&cmd_info[i], i);
          }else{
            cmd_info[i].handler();
          }
          break;
        }
      }
      
      if (i == num_cmds) {
        unknownCommand();
      }
      
      //Now we are done so print the prompt
      printPrompt();
    }
  }
}
