/****************************************************************************
*
*   Copyright (c) 2006 Carrick Detweiler
*                      and Massachusetts Institute of Technology
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*
*   $Id: queue.h 2872 2011-02-20 17:22:00Z carrick $
****************************************************************************/

/*****************USAGE********************
 * 
 * WARNING: Not interrupt safe!!! Enclose calls with:
 *   cli();
 *   QUEUE_...
 *   sei();
 * if used in an area of code where interrupts are currently enabled.
 *
 * to create a new queue:
 * QUEUE_NEW(RxQueue,size);
 *
 */

#if !defined( QUEUE_H )
#define QUEUE_H

volatile typedef struct {
  volatile unsigned char putIdx;
  volatile unsigned char size;
  volatile unsigned char maxSize;
  char *queue;
} queue_t;

/**
 * Creates a new FIFO queue with the specified name.  For example:
 * QUEUE_NEW(RxQueue,size);
 *
 * This is the only init that is needed.
 **/
#define QUEUE_NEW( name , size)                 \
  char name##_queue_real[(size)];            \
  queue_t name = {0,0,(size),name##_queue_real};

/**
 * Reinits the specified queue (basically clears it).
 **/
#define QUEUE_REINIT(q)   \
  {(q).putIdx=(q).size=0;}

/**
 * Removes and returns the last in element.
 **/
#define DEQUEUE( q ) \
  (((q).queue)[ ((q).putIdx - --(q).size + (q).maxSize) % ((q).maxSize) ])

/**
 * Returns the nth element without removing it from the queue.
 **/
#define QUEUE_PEAK( q , n )                                             \
  (((q).queue)[ (((q).putIdx + (n) - ((q).size-1)) + (q).maxSize) % ((q).maxSize) ])

/**
 * Adds element v to the queue.
 **/
#define ENQUEUE( q , v)            \
  {((q).queue)[ (q).putIdx = (++(q).putIdx % ((q).maxSize)) ] = (v);(q).size++;}

/**
 * Returns the number of elements in the queue.
 **/
#define QUEUE_SIZE( q ) \
  ((q).size)

/**
 * Gets the maximum number of elements the queue can hold.
 **/
#define QUEUE_MAX_SIZE( q ) \
  ((q).maxSize)

/**
 * Gets the number of free spaces in the queue.
 **/
#define QUEUE_FREE( q ) \
  (((q).maxSize) - ((q).size))

/**
 * Returns 1 if the queue is empty.
 **/
#define QUEUE_ISEMPTY( q ) \
  ((q).size <= 0)

/**
 * Returns 1 if the queue is full.
 **/
#define QUEUE_ISFULL( q ) \
  ((q).size >= (q).maxSize)

#endif /* QUEUE_H */
