/** Classifier API - Java Implementation
 * 
 * @author Christopher Hammack <chammack@cse.unl.edu>
 * @version 0.7 
 * 
 **/
package edu.unl.lasso.Learner.MyLearner;

public interface MyLearner {

   public class ClassifyResult {
      public float[] confidences;
      public String[] labels;
   }

   public class ExtractionResult {
   	  public float[] confidences;
   	  public String[][] fields;
   }

   /** Convenience function that allows classification initialization 
       to take place.  Also provides classifier name.

       @param Classifier Name       
   **/
   public void classifyStart(String name) throws java.lang.Exception;

   /** Convenience function that allows training initialization 
       to take place. Also provides classifier name.

       We do batch learning, so we will delete the old classifier
       data.

       @param Classifier Name
   **/
   public void trainStart(String name) throws java.lang.Exception;


   /** Convenience function that allows classification deinitialization 
       to take place.
       Many implementations may not need this


   **/
   public void classifyStop() throws java.lang.Exception; 

   /** Convenience function that allows training deinitialization 
       to take place.
       Some implementations may not need this.
    
       In a typical "batch training" session, this function will
       actually start the learning.  The train function would be
       used to accumulate the data and train_stop would act on 
       the accumulated data.

   **/
   public void trainStop() throws java.lang.Exception; 

   /** Option Attribute Function
       Classifiers may have various options for classification
       
    <P><B>LASSO Option Naming Convention:</B>
    <P>Options should be lower case with underscores
    separating them.  For example:  "kernel_type"
    rather than "kernelType".

    <P>A few LASSO built-in options you can receive if you'd like:
    <PRE><CODE>
    "attribute_count"  - gives the size of the feature vector (per example)
    "label_x" (where x is an integer >= 1) - the label name
    </CODE></PRE>


    @param attribute attribute to set
    @param value associated value of option
       
   **/

   public void setOption(String attribute, String value) throws java.lang.Exception; 
   


   /** Main Classification function.  
       Feature is passed in as a string, with specified length.
       
    Feature is passed in using appropriate representation,
    with specified length.

    Most common feature representation is CSV (Comma Separated Value).
    Feature Types:
    <TT>
        "1.254, 3.21, 4.0" (real valued features)
        or "0,0,1,1,1,0" (binary features)
        or "1,2,4,2,1,0" (integer features)
    </TT>

    In each of these cases, "features" is of type Float[].  You should
    typecast the Object[] to this:
    <TT>Float[] featureFloat = (Float[]) feature;</TT>
   
    However, "text" and "binary" features are also possible--
    although only pertinent to advanced "all-in-one"
    type learning algorithms.  The feature type is defined when
    registering your learning algorithm/classifier.

    Function should return labels and the corresponding
    confidences array through ClassifyResult.
    
    You should return a list of possible labels in this array
    with it's respected confidence in the same index.

    <P><B>Example:</B>
       <PRE><CODE>
       Given the classification result:
           Red (75% confidence)
           Blue (25% confidence)
           Green (25% confidence)
       You should effectively set:
       	   ClassifyResult cr = new ClassifyResult();
       	   cr.labels = new String[3];
       	   cr.confidences = new float[3];
           cr.labels[0] = new String("Red");
           cr.confidences[0] = 0.75;
           ...
		</CODE></PRE>
		
		@param feature Object representation of the feature
        @return ClassifyResult structure
   **/

   public ClassifyResult classify(Object[] feature) throws java.lang.Exception;

   /** Main Train function
       A feature is provided, along with a label.  The length of
       the feature text is also provided.

       This works very simply: write the feature to a unique filename
       in a directory specified by the label.
       
       @param feature Object representation of the feature
       @param label the label of the feature
   **/
   public void train(Object[] feature, String label) throws java.lang.Exception;

   /** Main Training function for extraction tasks
       A feature is provided, along with a list of attributes and
       an attribute list.
       @param feature Object[] representation of the feature
       @param attributes The attributes in string format
       @param attribcount number of attributes
   **/
   public void train(Object[] feature, String[] attributes, int attribcount) 
          throws java.lang.Exception;


   /** Extraction classification function
       Provide a feature of certain length.
       Returns an array of attributes (attribute and value) of 
	 length attribcount (returned by reference)
	 
	 @param feature Object[] representation of the feature
	 @return extraction result
   **/
   public ExtractionResult extract(Object[] feature) throws java.lang.Exception;


   /** Prepares the export of a classifier.
       Input is the name of the classifier, 
       Return is the path of the name of the _single bytestream_ to export
       
       @param name name of classifier
       @return classifier bytedata
   **/
   public byte[] exportClassifier(String name) throws java.lang.Exception;

   /** Imports the classifier from it's transmission format
       into the format expected locally.

       In this classifier that consists of extracting a tar file
	 and deleting the original.
	 @param name name of classifier
	 @param data byte data representating classifier
   **/
   public void importClassifier(String name, byte[] data) throws java.lang.Exception;


   /** Checks the age of the classifier
       This is used to determine if the local copy needs to be
       replaced or not by the copy on LASSO

       This should return a UNIX timestamp form time for the last
       modification of the classifier.  UNIX timestamp is seconds
       since 1970 (see "man time.2"). 

       In this classifier, the vocabulary file is used as it is
       always updated if the classifier is.
       @param name name of classifier
       @return age of classifier
   **/
   public long getClassifierAge(String name) throws java.lang.Exception;
}
    
    
