/** DumbLearner - Java Implementation
 * 
 * @author Christopher Hammack <chammack@cse.unl.edu>
 * @version 0.7 
 * 
 **/

import java.util.Random;
import java.util.Hashtable;
import java.util.Vector;
import java.io.*;
import edu.unl.lasso.Learner.MyLearner.*;

public class DumbLearner implements edu.unl.lasso.Learner.MyLearner.MyLearner, java.io.Serializable {

   private class ProbabilityData implements java.io.Serializable {
      public String[] labels;
      public float[] probabilities;
   }

   private Random rand; 
   private int seenExamples = 0;
   private Vector labelKeys = new Vector(); // temp, keeps keys in order
   private Hashtable counts = new Hashtable(); // temp hashtable
   private ProbabilityData myProbabilityData; 
   private String myName;

   /** Convenience function that allows classification initialization 
       to take place.  Also provides classifier name.

       @param Classifier Name       
   **/
   public void classifyStart(String name) throws java.lang.Exception {
      try {
         rand = new Random();
         FileInputStream fis = new FileInputStream(name + ".obj");
         ObjectInputStream ois = new ObjectInputStream(fis);
 
         myProbabilityData = (ProbabilityData) ois.readObject();
         ois.close();
         fis.close();
      }
      catch(Exception e) {
         e.printStackTrace();
         throw new java.lang.Exception("IOException");
      } 
   }

   /** Convenience function that allows training initialization 
       to take place. Also provides classifier name.

       We do batch learning, so we will delete the old classifier
       data.

       @param Classifier Name
   **/
   public void trainStart(String name) throws java.lang.Exception {
      myName = name;
   }


   /** Convenience function that allows classification deinitialization 
       to take place.
       Many implementations may not need this


   **/
   public void classifyStop() throws java.lang.Exception {
      return;
   }

   /** Convenience function that allows training deinitialization 
       to take place.
       Some implementations may not need this.
    
       In a typical "batch training" session, this function will
       actually start the learning.  The train function would be
       used to accumulate the data and train_stop would act on 
       the accumulated data.

   **/
   public void trainStop() throws java.lang.Exception {
      int labelCount = labelKeys.size();
      myProbabilityData = new ProbabilityData();
      myProbabilityData.labels = new String[labelCount];
      myProbabilityData.probabilities = new float[labelCount];

      for(int i = 0; i < labelCount; i++) {
         myProbabilityData.labels[i] = (String) labelKeys.get(i);
         Integer exampleCount = (Integer) counts.get(myProbabilityData.labels[i]);
         myProbabilityData.probabilities[i] 
             = (float) exampleCount.intValue()/(float) seenExamples;
      }

      try {
         FileOutputStream fos = new FileOutputStream(myName + ".obj");
         ObjectOutputStream oos = new ObjectOutputStream(fos);
         oos.writeObject(myProbabilityData);
      }
      catch(Exception e) {
         e.printStackTrace();
         throw new Exception("IOException");
      }
   }

   /** Option Attribute Function
       Classifiers may have various options for classification
       
    <P><B>LASSO Option Naming Convention:</B>
    <P>Options should be lower case with underscores
    separating them.  For example:  "kernel_type"
    rather than "kernelType".

    <P>A few LASSO built-in options you can receive if you'd like:
    <PRE><CODE>
    "attribute_count"  - gives the size of the feature vector (per example)
    "label_x" (where x is an integer >= 1) - the label name
    </CODE></PRE>


    @param attribute attribute to set
    @param value associated value of option
       
   **/

   public void setOption(String attribute, String value) throws java.lang.Exception {
      if(attribute.indexOf("label_") >= 0) {
         counts.put(value, new Integer(0)); // Initialize entry in hashtable
         labelKeys.add(value);
      }
   }    
   

   /** Main Classification function.  
       Feature is passed in as a string, with specified length.
       
    Feature is passed in using appropriate representation,
    with specified length.

    Most common feature representation is CSV (Comma Separated Value).
    Feature Types:
    <TT>
        "1.254, 3.21, 4.0" (real valued features)
        or "0,0,1,1,1,0" (binary features)
        or "1,2,4,2,1,0" (integer features)
    </TT>

    In each of these cases, "features" is of type Float[].  You should
    typecast the Object[] to this:
    <TT>Float[] featureFloat = (Float[]) feature;</TT>
   
    However, "text" and "binary" features are also possible--
    although only pertinent to advanced "all-in-one"
    type learning algorithms.  The feature type is defined when
    registering your learning algorithm/classifier.

    Function should return labels and the corresponding
    confidences array through ClassifyResult.
    
    You should return a list of possible labels in this array
    with it's respected confidence in the same index.

    <P><B>Example:</B>
       <PRE><CODE>
       Given the classification result:
           Red (75% confidence)
           Blue (25% confidence)
           Green (25% confidence)
       You should effectively set:
       	   ClassifyResult cr = new ClassifyResult();
       	   cr.labels = new String[3];
       	   cr.confidences = new float[3];
           cr.labels[0] = new String("Red");
           cr.confidences[0] = 0.75;
           ...
		</CODE></PRE>
		
		@param feature Object representation of the feature
        @return ClassifyResult structure
   **/

   public ClassifyResult classify(Object[] feature) throws java.lang.Exception {
      Float[] floatFeature = (Float[]) feature;

      // Flip a probability-biased coin
      // Generate the random number from 0->1
      float num = rand.nextFloat();
      float sum = (float) 0.0;
  
      int i = 0;
      for(i = 0; i < labelKeys.size(); i++) {
         if(num < (sum + myProbabilityData.probabilities[i]))
	    break;
	 sum += myProbabilityData.probabilities[i];
      }
   
      // Find the label which matches the index
      ClassifyResult cr = new ClassifyResult();
      // We only return a single label result
      cr.labels = new String[1];
      cr.confidences = new float[1];
      cr.labels[0] = myProbabilityData.labels[i];
      cr.confidences[0] = (float) 1.0;
      return cr; 
   }  	

   /** Main Train function
       A feature is provided, along with a label.  The length of
       the feature text is also provided.

       This works very simply: write the feature to a unique filename
       in a directory specified by the label.
       
       @param feature Object representation of the feature
       @param label the label of the feature
   **/
   public void train(Object[] feature, String label) throws java.lang.Exception {
      // Look for current label in hashtable, increment count
      Integer curCount = (Integer) counts.get(label);
      Integer newCount = new Integer(curCount.intValue() + 1);
      counts.put(label, newCount);
      seenExamples++;
   } 
       

   /** Main Training function for extraction tasks
       A feature is provided, along with a list of attributes and
       an attribute list.
       @param feature Object[] representation of the feature
       @param attributes The attributes in string format
       @param attribcount number of attributes
   **/
   public void train(Object[] feature, String[] attributes, int attribcount) 
          throws java.lang.Exception {
      throw new Exception("NotImplementedException");
      // DO NOT IMPLEMENT
   }



   /** Extraction classification function
       Provide a feature of certain length.
       Returns an array of attributes (attribute and value) of 
	 length attribcount (returned by reference)
	 
	 @param feature Object[] representation of the feature
	 @return extraction result
   **/
   public ExtractionResult extract(Object[] feature) 
          throws java.lang.Exception {
      throw new Exception("NotImplementedException");
      // DO NOT IMPLEMENT
   } 


   /** Prepares the export of a classifier.
       Input is the name of the classifier, 
       Return is the path of the name of the _single bytestream_ to export
       
       @param name name of classifier
       @return classifier bytedata
   **/
   public byte[] exportClassifier(String name) throws java.lang.Exception {
       return new byte[1];
       // DO NOT IMPLEMENT
   }

   /** Imports the classifier from it's transmission format
       into the format expected locally.

       In this classifier that consists of extracting a tar file
	 and deleting the original.
	 @param name name of classifier
	 @param data byte data representating classifier
   **/
   public void importClassifier(String name, byte[] data) 
       throws java.lang.Exception {
      return;
      // DO NOT IMPLEMENT
   }  


   /** Checks the age of the classifier
       This is used to determine if the local copy needs to be
       replaced or not by the copy on LASSO

       This should return a UNIX timestamp form time for the last
       modification of the classifier.  UNIX timestamp is seconds
       since 1970 (see "man time.2"). 

       In this classifier, the vocabulary file is used as it is
       always updated if the classifier is.
       @param name name of classifier
       @return age of classifier
   **/
   public long getClassifierAge(String name) throws java.lang.Exception {
      return 0;
      // DO NOT IMPLEMENT
   }
}
    
    
