/* LearnerWrapper -
   Provides a command line interface to Lasso learner
   components

   Files:
      filename.train - training data
      filename.test - test data
      filename.names - feature specifications

   The current train/test parser expects space delimited
   data.  You will likely need to change this.  It is also
   hardcoded for a certain number and type of feature
   vector.  However, much of the work is done for you
   (look at the DataFormat structure)--you just need to
   exploit it to get dynamic data format handling.

   This wrapper could be more efficiently implemented
   using java datastructures, but is written as-is to
   maintain similarity to the C++ version.
*/

/**
   @author Christopher Hammack <chammack@cse.unl.edu>
**/

import java.io.*;
import java.util.StringTokenizer;
import java.util.Vector;

public class LearnerWrapper {

   private class Example {
      Float[] data;
      String label;
   }

   private class FeatureFormat {
      int featureType;
      Vector attributeValues;       
   }
 
   private class DataFormat {
      FeatureFormat[] features;
   }


   // Global Constants
   public static final int TYPE_TEXT = 1;
   public static final int TYPE_NUMERIC = 2;
   public static final int TYPE_LABEL = 3;

   public static void main(String args[]) {
      MyLearner learner = new DumbLearner();

      if(args.length < 4) {
	 System.out.println("Usage: java LearnerWrapper action filename feature_count label_count");
	 System.out.println("Where:");
	 System.out.println("\t action - action to perform: 'test' or 'train'"); 
	 System.out.println("\t filename - file to use for test or train"); 
	 System.out.println("\t feature_count - number of features"); 
	 System.out.println("\t label_count - number of labels"); 
	 System.exit(1); 
      }

      if(!args[0].equals("train") && !args[0].equals("test"))
      {
	 System.out.println("Action must be either 'train' or 'test'");
	 System.exit(1);
      }

      int featureCount = Integer.parseInt(args[2]);

      // Read names file
     
      LearnerWrapper lw = new LearnerWrapper(); 
      String filename =  args[1] + ".names";
      DataFormat format = lw.loadNamesFile(filename, featureCount);

      // Send the learner some information
      String featureVectorLength =  ""+ featureCount;
      try {
         learner.setOption("attribute_count", featureVectorLength);  

	 // loop over labels (assume last value is a label)
	 int vals = format.features[featureCount].attributeValues.size();
	 for(int i = 0; i < vals ; i++) {
	    String ptr = (String) format.features[featureCount].attributeValues.get(i);
	    learner.setOption("label_" + i, ptr);
	 } 
      }
      catch(Exception e) {
         e.printStackTrace();
         System.exit(1);
      }

      // Start Training Code
      if(args[0].equals("train")) {
         FileReader fr = null;
         BufferedReader infile = null;
         try { 
	    learner.trainStart(args[1]); 
	    fr = new FileReader(args[1] + ".train");
	    infile = new BufferedReader(fr);
         }
         catch(Exception e) {
	    System.out.println("Error: unable to open train file");
	    System.exit(1);
	 }
    
	 Example train_example = lw.readNextExampleFromFile(infile, format);
	 while(train_example != null) {
            try {
	       learner.train(train_example.data, train_example.label);
            }
            catch(Exception e) {
               e.printStackTrace();
            }
	    train_example = lw.readNextExampleFromFile(infile, format);
	 }
         try {
	    infile.close();
	    learner.trainStop();   
         }
         catch(Exception e) { 
            e.printStackTrace();
         }

      }
      else if(args[0].equals("test")) { 
         FileReader fr = null;
         BufferedReader infile = null;
         try { 
	    learner.classifyStart(args[1]);
	    fr = new FileReader(args[1] + ".test");
	    infile = new BufferedReader(fr);
         }
         catch(Exception e) {
	    System.out.println("Error: unable to open test file");
	    System.exit(1);
	 }

	 Example test_example = lw.readNextExampleFromFile(infile, format);
	 while(test_example != null)
	 {
            try {
	       MyLearner.ClassifyResult cr = learner.classify(test_example.data);
	       // Print Result
	       System.out.println("Classification result:");
	       System.out.println("Possible labels: " + cr.labels.length);
	       System.out.println("Top Label: "  + cr.labels[0] +"\tConfidence:" +cr.confidences[0]);
	       test_example = lw.readNextExampleFromFile(infile, format);
            }
            catch(Exception e) {
               e.printStackTrace();
               System.exit(1);
            }

	 }
         try { 
	    infile.close();
	    learner.classifyStop();
         }
         catch(Exception e) {
            e.printStackTrace();
         }
      }              

   }

   /* Reads an example from file based on DataFormat df. */
   Example readNextExampleFromFile(BufferedReader infile, DataFormat df) {
      try {
	 String line;
	 if(infile.ready()) {
	    Example the_example = new Example();
	    the_example.data = new Float[df.features.length];

	    line = infile.readLine();

	    if(line==null)
	       return null;

	    // TODO: You should handle this dynamically, this is just
	    //  provided as an example.  
	    //  Hardcoded example: features are string value string label
	    // HINT: use DataFormat
       
	    // sky
	    StringTokenizer strtok = new StringTokenizer(line, " ");
	    String token = strtok.nextToken();
	    the_example.data[0] = (Float) new Float(translateStringToNumeric(token, 0, df));

	    // temp (already numeric--do not map)
	    token = strtok.nextToken(); 
	    the_example.data[1] = new Float(Float.parseFloat(token));

	    // humidity
	    token = strtok.nextToken();
	    the_example.data[2] = (Float) new Float(translateStringToNumeric(token, 2, df));

	    // label
	    token = strtok.nextToken();
	    the_example.label = token;
	
	    return the_example;
	 }
      }
      catch(Exception e) {
         e.printStackTrace();
      } 
	 
      return null;
		      
   }

   /* Translate a string attribute into a numeric one
      based on the names file 
   */
   private int translateStringToNumeric(String value, int featurePosition, 
       DataFormat df) {
   
       return df.features[featurePosition].attributeValues.indexOf(value);
   }

   /* Parse the names file */   
   private DataFormat loadNamesFile(String filename, int featureCount) {
      DataFormat format = new DataFormat();
      FileReader fr = null;
      BufferedReader br = null;

      try {
         fr = new FileReader(filename);
         br = new BufferedReader(fr);
      } 
      catch(Exception e) { 
	 System.out.println("Cannot find names file: " + filename);
	 System.exit(1);
      }
     
      format.features = new FeatureFormat[featureCount+1];
      try {
	 String line;
	 int i = 0;
	 while(br.ready()) {
	    line = br.readLine();
	    StringTokenizer strtok = new StringTokenizer(line, " ");
	    String token = strtok.nextToken();
            format.features[i] = new FeatureFormat();
            format.features[i].attributeValues = new Vector();
 
	    if(token.equals("text"))
	       format.features[i].featureType = TYPE_TEXT;
	    else if(token.equals("numeric"))
	       format.features[i].featureType = TYPE_NUMERIC;
	    else if(token.equals("label"))
	       format.features[i].featureType = TYPE_LABEL;

	    while(strtok.hasMoreTokens())  {
	       token = strtok.nextToken();
	       format.features[i].attributeValues.add(token);
	    }
	    i++;
      }
      return format;
    }
    catch(Exception e) {
       e.printStackTrace();
       System.exit(1);
    }
    return null; 
  }
 
}
